
#include "MK64F12.h"
#include "platform.h"
#include "i2s.h"


/* Name: i2s_init   
* Description: Initialization I2S bus clock and Pins.   
* Pins ready to interact with the QSB BASE BOARD
* I2S0 TXD0     PTC1
* I2S0 RXD0     PTC5
* I2S0 RX_BCLK  PTC.9
* I2S0 RX_FS    PTC.7
*
* Configuration: Slave, 16 bits data, Stereo
*/    

void i2s_init() {   
  
SIM->SCGC5 |=  SIM_SCGC5_PORTC_MASK ; 

PORTC->PCR[1] = PORT_PCR_MUX(6) ;       // I2S0 TXD0 (Alt 6) PTC1
PORTC->PCR[5]  = PORT_PCR_MUX(4) ;      // I2S0 RXD0 (Alt 4) PTC5
	
PORTC->PCR[9]  = PORT_PCR_MUX(4) ;      // I2S0 RX_BCLK (Alt 4) PTC.9

PORTC->PCR[7] = PORT_PCR_MUX(4) ;   // I2S0 RX_FS (Alt 4) PTC.7
	
SIM->SCGC6 |=SIM_SCGC6_I2S_MASK;        // enable clock to the I2S module
I2S0->MCR  &= ~(I2S_MCR_MOE_MASK);      // Master clock pin is configured  
                                        // as an input.	
		
/* I2S is configured to work with 32-bits frame that contains two 16-bits samples. The CODEC is configures in DSP mode. It will generate short frame sync pulse followed by two 16-bits samples at fixed bit clock rate (3.04MHz). */


	
	I2S0->TCR2 |= I2S_TCR2_SYNC(01);      // Use synchronous mode (Use the rx BCLCK and rx WCLK)
  I2S0->TCR2 &= ~(I2S_TCR2_BCD_MASK);   // Do not generate bit clock
  // bit clock is generated externally in slave mode
  I2S0->TCR3 |= I2S_TCR3_TCE(1);        // Enable transmit to data channel 0
  I2S0->TCR4 |= I2S_TCR4_FRSZ(0);       // Frame size in words = 1
  I2S0->TCR4 |= I2S_TCR4_SYWD(31);      // Bit width of WCLK (32 bits-1)
  I2S0->TCR4 |= I2S_TCR4_MF_MASK;       // MSB First
  I2S0->TCR4 &= ~(I2S_TCR4_FSD_MASK);   // Not generate WCLK, slave mode
  I2S0->TCR4 &= ~(I2S_TCR4_FSE_MASK);   // No extra bit before frame starts
  I2S0->TCR5 |= I2S_TCR5_W0W(31);       // Bits per word, first frame
  I2S0->TCR5 |= I2S_TCR5_WNW(31);       // Bits per word, nth frame
  I2S0->TCR5 |= I2S_TCR5_FBT(31);       // Index shifted for FIFO
	I2S0->TCSR |= I2S_TCSR_DBGE_MASK;     // Enable transmit operatioin in Debug mode
	
	I2S0->RCR2 |= I2S_RCR2_SYNC(00);      // Use asynchronous mode (recive the BCLCK and WCLK externaly)
  I2S0->RCR2 &= ~(I2S_RCR2_BCD_MASK);   // Bit clock direction. Input
  //Bit clock is generated externally in slave mode.
  I2S0->RCR3 |= I2S_RCR3_RCE(1);        // Enable recive to data channel 0
  I2S0->RCR4 |= I2S_RCR4_FRSZ(0);       // Frame size in words = 1
  I2S0->RCR4 |= I2S_RCR4_SYWD(31);      // Bit width of WCLK (32 bits-1)
  I2S0->RCR4 |= I2S_RCR4_MF_MASK;       // Most significant bit first
  I2S0->RCR4 &= ~(I2S_RCR4_FSD_MASK);   // Not generate WCLK, slave mode
  I2S0->RCR4 |= (I2S_TCR4_FSE_MASK);    // Extra bit before frame starts
  I2S0->RCR5 |= I2S_RCR5_W0W(31);       // Bits per word, first frame
  I2S0->RCR5 |= I2S_RCR5_WNW(31);       // Bits per word, nth frame
  I2S0->RCR5 |= I2S_RCR5_FBT(31);       // Index shifted for FIFO
	I2S0->RCSR |= I2S_RCSR_DBGE_MASK;     // Enable receiver operatioin in Debug mode
}

void i2s_rx_irq_enable(){
  I2S0->RCSR |= I2S_RCSR_FRIE_MASK;     // FIFO request interrupt enable
}

void i2s_tx_irq_enable(){
  I2S0->TCSR |= I2S_TCSR_FRIE_MASK;     // FIFO request interrupt enable 
}

void i2s_rx_depth_irq(int d){
  I2S0->RCR1 |= I2S_RCR1_RFW(d);        // Set FIFO watermark
}

void i2s_tx_depth_irq(int d){
  I2S0->TCR1 |= I2S_TCR1_TFW(d);        // Set FIFO watermark 
}

void i2s_rx_dma1_enable(){
  I2S0->RCSR |= I2S_RCSR_FRDE_MASK;  //Enable I2S receive DMA request
}

void i2s_tx_dma1_enable(){
  I2S0->TCSR |= I2S_TCSR_FRDE_MASK;     //Enable I2S transmit DMA request
}

void i2s_rx_depth_dma1(int d){
  I2S0->RCR1 |= I2S_RCR1_RFW(d);        // Set FIFO watermark
}

void i2s_tx_depth_dma1(int d){
  I2S0->TCR1 |= I2S_TCR1_TFW(d);        // Set FIFO watermark 
}

void i2s_rx_dma2_enable(){
  I2S0->RCSR |= I2S_RCSR_FRDE_MASK;     //Enable I2S receive DMA request	 
}

void i2s_tx_dma2_enable(){
 I2S0->TCSR |= I2S_TCSR_FRDE_MASK;      //Enable I2S transmit DMA request   
}

void i2s_rx_depth_dma2(int d){
  I2S0->RCR1 |= I2S_RCR1_RFW(d);        // Set FIFO watermark
}

void i2s_tx_depth_dma2(int d){
  I2S0->TCR1 |= I2S_TCR1_TFW(d);        // Set FIFO watermark 
}

void i2s_start(void){
	
	I2S0->RCSR |= I2S_RCSR_FR_MASK;       // Global transmit enable 
  I2S0->RCSR |= I2S_RCSR_RE_MASK;       // Global transmit enable 
  I2S0->TCSR |= I2S_TCSR_FR_MASK;       // Global transmit enable 
  I2S0->TCSR |= I2S_TCSR_TE_MASK;       // Global transmit enable 
}

void i2s_stop(void){
    //Clear STOP,RESET and MUTE bit
  I2S0->TCSR &= ~I2S_TCSR_TE_MASK;      // Global transmit enable 
  I2S0->RCSR &= ~I2S_RCSR_RE_MASK;      // Global receive enable
}

uint32_t i2s_rx(void){
return 	I2S0->RDR[0];	//Reads data from the data register  
}

void i2s_tx(uint32_t c){
I2S0->TDR[0] = c;  
while ((I2S0->TCSR & I2S_TCSR_FWF_MASK)==1 ){}  //Wait for the FIFO Request Flag to be clean

}

static void (*i2s_callback)(void) = 0;

void i2s_set_callback(void (*callback)(void)) {
	i2s_callback = callback;


   NVIC_EnableIRQ(I2S0_Tx_IRQn);
   NVIC_ClearPendingIRQ( I2S0_Tx_IRQn);
	
}

 
 void I2S0_Tx_IRQHandler(void) {  //TX and RX HANDLER happen at same time, so we will do both inside the same interruption
	if (i2s_callback) {
		i2s_callback();
	}
}


// *******************************ARM University Program Copyright © ARM Ltd 2014*************************************   
