
#include "MK64F12.h"
#include "platform.h"
#include "delay.h"
#include "i2c.h"

//I2C write o read
#define WRITE 0x00 // Master write     
#define READ 0x01 // Master read 

/* Name: i2c_init   
* Description: Initialization I2C bus clock and Pins.   
* Transmition frequency 100KHz. 
* I2C0 SDA P1_30
* I2C0_SCK P1_31
*/    

void i2c_init() {   

  SIM->SCGC4 |= SIM_SCGC4_I2C1_MASK;   //Turn on clock to I2C0 module.   
  SIM->SCGC5 |= SIM_SCGC5_PORTC_MASK;  //Turn on Port C
  // Configure GPIO’s     
  PORTC->PCR[10] = PORT_PCR_MUX(2) | PORT_PCR_ODE_MASK ; //PTC.10 is I2C_DATA -> I2C1_SCL(alt2)
  PORTC->PCR[11] = PORT_PCR_MUX(2) | PORT_PCR_ODE_MASK ; //PTC.11 is I2C_CLOCK ->I2C1_SDA(alt2)
  //Both pins are configured in Open Drain mode. 
  //The audio board have the convenient pull-up resistors.
	I2C1->F = 0x6D; //Configure the I2C clock to 100kHz (120000khz/( mul=2,RCL=640))
	I2C1->C1 |= I2C_C1_IICIE_MASK;   //enable interrupt I2C
  I2C1->C1 |= I2C_C1_IICEN_MASK;   //enable I2C  
	
}
/* Name: i2c_start  
* Description: Make the start signal (master)
* The start signal is automatically generated when the master 
* enables transmissions.
*/  
void i2c_start(){

I2C1->C1 |= I2C_C1_TX_MASK;  //Enable transmissions
I2C1->C1 |= I2C_C1_MST_MASK; //Configure microcontroller as a master.
}

/* Name: i2c_stop  
* Description: Make the stop signal (master)
* The stop signal is automatically generated when the master 
* disable any transmissions.
*/  
void i2c_stop(){

I2C1->C1 &= ~I2C_C1_MST_MASK;  //Disable master.
I2C1->C1 &= ~I2C_C1_TX_MASK;   //Disable transmissions.
delay_ms(1);                      //Wait until Stop signal has been generated.
}

/* Name: i2c_wait     
* Description: waits until 8 bits of data has been transmitted or received  
*/    
void i2c_wait() {    
  while((I2C1->S & I2C_S_IICIF_MASK)==0) {} 
  //This flag indicates the one byte transmission is completed.  
  I2C1->S |= I2C_S_IICIF_MASK;    // Clear the interrupt flag (1) 
}      

/* Name: i2c_tx     
* Description: Puts data in de data register for send.  
*/
void i2c_tx(char data){
I2C1->D = data;  
}

/* Name: i2c_rx     
* Description: Puts data in de data register for send.  
*/
unsigned char i2c_rx(void){
return I2C1->D;  
}

/* Name: i2c_write    
* Description: Write buff_len number of bytes to the addressed slave device. 
*/
void i2c_write(uint8_t address, uint8_t *buffer, int buff_len) {
 int i = 0;
 i2c_start();
 address = (address << 1)| WRITE;  //7 bits address and one 0 to indicate we want to write on de CODEC 
 i2c_tx(address );
 i2c_wait();
 for (i = 0; i < buff_len; i++) {
  i2c_tx(buffer[i]);
	i2c_wait();
 }
 i2c_stop();
}

/* Name: i2c_read    
* Description:Read buff_len number of bytes from the addressed slave device. 
*/
void i2c_read(uint8_t address, uint8_t *buffer, int buff_len) {
 int i = 0;
 i2c_start();
 i2c_tx(address | READ);
 for (i = 0; i < buff_len - 1; i++) {
	i2c_rx(); 
  buffer[i] = i2c_rx(); ;
  I2C1->C1 &=  (0 << I2C_C1_TXAK_SHIFT); //ack

 }
 buffer[i] = i2c_rx();
 I2C1->C1 &=  ~(1 << I2C_C1_TXAK_SHIFT); //nack  
 i2c_stop();
}


// *******************************ARM University Program Copyright © ARM Ltd 2014*************************************   
