

#ifndef audio_h
#define audio_h

#include "MK64F12.h"
#include <gpio.h>
#include <delay.h>
#include "i2c.h"
#include "i2s.h"
#include "dma.h"
#include "audio.h"

	
void init_LED() {  // init leds
gpio_set_mode(LEDR, Output);
gpio_set_mode(LEDG, Output);
gpio_set_mode(LEDB, Output);
gpio_set(LEDR, HIGH);
gpio_set(LEDG, HIGH);
gpio_set(LEDB, HIGH);
}

typedef struct BITVAL    // used in function prbs()
{
 unsigned short b0:1, b1:1, b2:1, b3:1, b4:1, b5:1, b6:1;
 unsigned short b7:1, b8:1, b9:1, b10:1, b11:1, b12:1,b13:1;
 unsigned short dweebie:2; //Fills the 2 bit hole - bits 14-15
} bitval;

typedef union SHIFT_REG
{
 unsigned short regval;
 bitval bt;
} shift_reg;
int fb = 1;                        // feedback variable
shift_reg sreg = {0xFFFF};         // shift register

short prbs(void) 			       //gen pseudo-random sequence {-1,1}
{
  int prnseq;
  if(sreg.bt.b0)
	prnseq = -NOISELEVEL;		           //scaled negative noise level
  else
	prnseq = NOISELEVEL;		           //scaled positive noise level
  fb =(sreg.bt.b0)^(sreg.bt.b1);   //XOR bits 0,1
  fb^=(sreg.bt.b11)^(sreg.bt.b13); //with bits 11,13 -> fb
  sreg.regval<<=1;
  sreg.bt.b0=fb;			       //close feedback path
  return prnseq;			       //return noise sequence value
}


/* Name: I2C_WriteRegister  
* Requires: Device Address, Device Register address, Data for register  
* Description: Writes the data to the device's register  
*/    
void Codec_WriteRegister ( uint32_t RegisterAddr, uint16_t RegisterValue) {    

uint8_t Byte[6];	
	
	// WM5102 has 32-bit address and 16-bit data
Byte[0] = (RegisterAddr>>24) & 0xFF;
Byte[1] = (RegisterAddr>>16) & 0xFF;
Byte[2] = (RegisterAddr>>8) & 0xFF;
Byte[3] = (RegisterAddr) & 0xFF;
Byte[4] = (RegisterValue>>8) & 0xFF;
Byte[5] = (RegisterValue) & 0xFF;
  delay_ms(1);  
	
i2c_write(CODEC_DEVICE_ADDRESS, Byte, 6);

}     
  
void configure_codec(uint16_t fs, int select_input)
{
	Codec_WriteRegister(0x0000, 0x0000);   // reset codec
  delay_ms(10);        

  Codec_WriteRegister(0x0019, 0x0001);   // patch codec (supplied by Wolfson)  
	Codec_WriteRegister(0x0080, 0x0003);
	Codec_WriteRegister(0x0081, 0xE022);
	Codec_WriteRegister(0x0410, 0x6080);
	Codec_WriteRegister(0x0418, 0xa080);
	Codec_WriteRegister(0x0420, 0xa080);
	Codec_WriteRegister(0x0428, 0xE000);
	Codec_WriteRegister(0x0443, 0xDC1A);
	Codec_WriteRegister(0x04B0, 0x0066);
	Codec_WriteRegister(0x0458, 0x000B);
	Codec_WriteRegister(0x0212, 0x0000);
	Codec_WriteRegister(0x0171, 0x0000);
	Codec_WriteRegister(0x035E, 0x000C);
	Codec_WriteRegister(0x02D4, 0x0000);
	Codec_WriteRegister(0x0080, 0x0000);  // end of patch

	Codec_WriteRegister(0x192, 0x8008);   // FLL2 12.000 MHz MCLK1 -> 24.576 MHz SYSCLK
	Codec_WriteRegister(0x193, 0x0018);   // could make this part of a switch in order
	Codec_WriteRegister(0x194, 0x007D);   // to allow 44.1kHz-related sample rates
	Codec_WriteRegister(0x195, 0x0008);
	Codec_WriteRegister(0x196, 0x0000);
	Codec_WriteRegister(0x191, 0x0001);

  delay_ms(1);                   

	Codec_WriteRegister(0x101, 0x0245);   // clock set up as SYSCLK = 24.576 MHz, src FLL2, SYSCLK_FRAC = 0
	                                      // SYSCLK enabled

	switch(fs) // set sample rate 1 - currently only 48kHz-related sample rates allowed
	{
		case hz8000:
			Codec_WriteRegister(0x102, 0x11);
			Codec_WriteRegister(0x580, 0x26);
		  break;
		case hz16000:
			Codec_WriteRegister(0x102, 0x12);
			Codec_WriteRegister(0x580, 0x28);
		  break;
		case hz24000:
			Codec_WriteRegister(0x102, 0x02);
			Codec_WriteRegister(0x580, 0x29);
		  break;
		case hz32000:
			Codec_WriteRegister(0x102, 0x13);
			Codec_WriteRegister(0x580, 0x2A);
		  break;
		case hz48000:
			Codec_WriteRegister(0x102, 0x03);
			Codec_WriteRegister(0x580, 0x2B);
		  break;
		default:
			Codec_WriteRegister(0x102, 0x11); // default is 8kHz sample rate
			Codec_WriteRegister(0x580, 0x26);
		  break;
	}

	Codec_WriteRegister(0x458, 0x0009);   // output noise gate enabled, threshold -84dB 

	Codec_WriteRegister(0x200, 0x0001);   
                                        
	Codec_WriteRegister(0x210, 0x00D5);   

  Codec_WriteRegister(0x584, 0x0002);   // AIF3 DSP mode A format: 0x584, 0x0000 
  Codec_WriteRegister(0x582, 0x0005);   // AIF3 LRCLK master 
 
	delay_ms(1);                  

  Codec_WriteRegister(0x587, 0x1010);   // AIF3 TX WL and SLOT_LEN both 16-bit
	Codec_WriteRegister(0x588, 0x1010);   // AIF3 RX WL and SLOT_LEN both 16-bit
	Codec_WriteRegister(0x59A, 0x0003);   // enable AIF3 RX channels (L and R)
	Codec_WriteRegister(0x599, 0x0003);   // enable AIF3 TX channels (L and R)
	Codec_WriteRegister(0x585, 0x0020);   // AIF3 32 BCLK cycles per LRC TX frame
	Codec_WriteRegister(0x586, 0x0020);   // AIF3 32 BCLK cycles per LRC RX frame

                                      // LINE OUT and HP OUT enabled in parallel 
  Codec_WriteRegister(0x690, 0x0030); // OUT2L (LINE OUT) mixer input is AIF3 RX1 (from I2S) 30
  Codec_WriteRegister(0x691, 0x0080); // associated volume is 0dB

  Codec_WriteRegister(0x698, 0x0031); // OUT2R (LINE OUT) mixer input is AIF3 RX2 (from I2S) 31
  Codec_WriteRegister(0x699, 0x0080); // associated volume is 0dB

  Codec_WriteRegister(0x680, 0x0030); // OUT1L (HP OUT) mixer input is AIF3 RX1 (from I2S)
  Codec_WriteRegister(0x681, 0x0080); // associated volume is 0dB

  Codec_WriteRegister(0x688, 0x0031); // OUT1R (HP OUT) mixer input is AIF3 RX2 (from I2S)
  Codec_WriteRegister(0x689, 0x0080); // associated volume is 0dB

// route LHPFs to AIF3TX
  Codec_WriteRegister(0x0780, 0x0060); // AIF3TX mixer from LHPF1 60
	Codec_WriteRegister(0x0788, 0x0061); // AIF3TX mixer from LHPF2 61
  Codec_WriteRegister(0x0781, 0x0080); // AIF3TX mixer gain 0dB
	Codec_WriteRegister(0x0789, 0x0080); // AIF3TX mixer gain 0dB

  Codec_WriteRegister(0x0EC0, 0x0003); // LHPF1 HPF enabled
  Codec_WriteRegister(0x0EC1, 0xF09e); // LHPF1 cutoff frequency in Hz depends on fs
  Codec_WriteRegister(0x0EC4, 0x0003); // LHPF2 HPF enabled
  Codec_WriteRegister(0x0EC5, 0xF09E); // LHPF2 cutoff frequency in Hz depends on fs

  Codec_WriteRegister(0x0901, 0x0080); // LHPF1 mixer source 1 gain 0dB
  Codec_WriteRegister(0x0909, 0x0080); // LHPF2 mixer source 1 gain 0dB

switch(select_input)
{
	case line_in:
    Codec_WriteRegister(0x0900, 0x0014); // LHPF1 mixer from IN3 (LINE IN)
	  Codec_WriteRegister(0x0908, 0x0015); // LHPF2 mixer from IN3
  	Codec_WriteRegister(0x0300, 0x0030); // enable IN3L and IN3R 0030 LINE IN
	  Codec_WriteRegister(0x0320, 0x2290); // IN3L PGA gain +8.0dB LINE IN (potential divider comp.)
	  Codec_WriteRegister(0x0321, 0x0280); // IN3L ADC volume 0dB LINE IN
	  Codec_WriteRegister(0x0324, 0x0090); // IN3R PGA gain +8.0dB LINE IN (potential divider comp.)
	  Codec_WriteRegister(0x0325, 0x0280); // IN3R ADC volume 0dB LINE IN
  break;
	case dmic_in:
    Codec_WriteRegister(0x0900, 0x0012); // LHPF1 mixer from IN2 (DMIC IN)
  	Codec_WriteRegister(0x0908, 0x0013); // LHPF2 mixer from IN2
    Codec_WriteRegister(0x0300, 0x000C); // enable IN2L and IN2R 000C DMIC IN
	  Codec_WriteRegister(0x0318, 0x3480); // IN2 DMIC IN IN2L PGA vol 0dB
	  Codec_WriteRegister(0x031D, 0x0280); // IN2R ADC volume 0dB DMIC IN
	  Codec_WriteRegister(0x0319, 0x0280); // IN2L ADC volume 0dB DMIC IN
	  Codec_WriteRegister(0x0219, 0x01A7); // MICBIAS2 enable DMIC IN
  break;
	case mic_in:
    Codec_WriteRegister(0x0900, 0x0010); // LHPF1 mixer from IN1 (MIC IN)
  	Codec_WriteRegister(0x0908, 0x0011); // LHPF2 mixer from IN1
  	Codec_WriteRegister(0x0300, 0x0003); // enable IN1L and IN1R 0003 MIC IN
	  Codec_WriteRegister(0x0310, 0x2A80); // IN1L PGA vol 0dB MIC IN
	  Codec_WriteRegister(0x0314, 0x0080); // IN1R PGA volume 0dB MIC IN
	  Codec_WriteRegister(0x0311, 0x0280); // IN1L ADC volume 0dB MIC IN
	  Codec_WriteRegister(0x0315, 0x0280); // IN1R ADC volume 0dB DMIC IN
	  Codec_WriteRegister(0x0218, 0x01A7); // MICBIAS1 enable MIC IN
  break;
	default:
    Codec_WriteRegister(0x0900, 0x0014); // LHPF1 mixer from IN3 (LINE IN)
	  Codec_WriteRegister(0x0908, 0x0015); // LHPF2 mixer from IN3
  	Codec_WriteRegister(0x0300, 0x0030); // enable IN3L and IN3R 0030 LINE IN
	  Codec_WriteRegister(0x0320, 0x2290); // IN3L PGA gain +8.0dB LINE IN (potential divider comp.)
	  Codec_WriteRegister(0x0321, 0x0280); // IN3L ADC volume 0dB LINE IN
	  Codec_WriteRegister(0x0324, 0x0090); // IN3R PGA gain +8.0dB LINE IN (potential divider comp.)
	  Codec_WriteRegister(0x0325, 0x0280); // IN3R ADC volume 0dB LINE IN
  break;
}		
 	Codec_WriteRegister(0x419, 0x0280); // DAC 2 volume L 0dB (LINE OUT)
	Codec_WriteRegister(0x41D, 0x0280); // DAC 2 volume R 0dB (LINE OUT)
	Codec_WriteRegister(0x411, 0x0280); // DAC 1 volume L 0dB (HP OUT)
	Codec_WriteRegister(0x415, 0x0280); // DAC 1 volume R 0dB (HP OUT)

	Codec_WriteRegister(0x400, 0x000F); // enable outputs OUT2L, OUT2R, OUT1L, OUT1R
}

/*--------initialization function--------------------------------
* Name: audio_init  
* 
* Configure the freescale and the CODEC 
*
* Requires: 
*
*   -Sampling rate; hz8000,  hz16000, hz24000, hz32000, hz48000,
*   -Input; line_in, mic_in dmic_in;
*   -Mode; inter, dma 
* 
* Description: 
* Configure the I2C protocol
* Write with I2C protocol into the CODEC registers  to configure it. 
* Configure the I2S protocol
* Configure the interruptions for the I2S RX FIFO and TX FIFO
*/ 
void audio_init ( char sampling_rate, char audio_input, char mode, void (*handler)(void)) {    
 
init_LED();
gpio_set_mode(PTC8, Output);	
i2c_init();   //Configure the I2C module 
gpio_set(PTC8,HIGH); //Pi board reset high
	
configure_codec(sampling_rate, audio_input); //Configure the WM5102 CODEC
	
i2s_init();   //Configure I2S
	
if (mode == dma){
 	
	  dma_init();
    dma_enable(0);
	  dma_enable(1);
	  
    dma_set_callback(handler);	
	
	  i2s_tx_depth_dma1(4);  //FIFO level on wich a TX irq request is created
    i2s_rx_depth_dma1(1);  //FIFO level on wich a RX irq request is created
    i2s_tx_dma1_enable();  //Enable I2S Transmit Interrupt
    i2s_rx_dma1_enable();  //Enable I2S Receive Interrupt
	  i2s_start();

  }
	
else{ //Mode interruptions
  
   i2s_tx_depth_irq(1);  //FIFO level on wich a TX irq request is created
   i2s_tx_irq_enable();  //Enable I2S Transmit Interrupt
	
	 i2s_start();
	 i2s_set_callback(handler);
		
}
}

#endif //audio

// *******************************ARM University Program Copyright © ARM Ltd 2014*************************************   
