
#include "STM32F4xx.h"
#include "LEDs.h"



/*----------------------------------------------------------------------------
 *--------------------------Application Note----------------------------------*
 Since power off can be swift and in this case it is hard to observe the 
 Interrupt. For demonstration, it is suggested that we can simulate the power
 off process by connecting a trimpot between ldd.
 
 Remove the Jumper JP1, this is usually used to measure the consumption of the 
 Core chip. Connect the trimpot and make sure it is turned to 0 ohm before next
 step. Then load the code to the board. You can see the orange LED is blinking
 at this point. Then slowly turn the trimpot, at around 22 ohm, the voltage 
 will then drop to the point that low enough to trigger the analog watchdog but
 not low enough to completely defunct the MCU. You will now be able to see the 
 red LED blinking (in a dimmed way). 
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
  Intializing the ADC1 with Vrefint as input (12-bit precision)
	Also setup the interrupt for ADC
 *----------------------------------------------------------------------------*/
void Init_ADC1(void){
	
	//Enable the clock for ADC module
	RCC->APB2ENR|=RCC_APB2ENR_ADC1EN;
	
	
	//Set the prescaler for the clock
	RCC->CFGR|=RCC_CFGR_PPRE2_DIV2;
	ADC->CCR|=ADC_CCR_ADCPRE_0;

  //Power up the ADC module
	ADC1->CR2|=ADC_CR2_ADON;
	
	//Use the Continous mode, so do not have to trigger everytime
	ADC1->CR2|=ADC_CR2_CONT;
	
	//480	cycles, better accuracy than 3 cycles
	ADC1->SMPR1|=ADC_SMPR1_SMP16;
	
	//Enable the Vrefint and ADC
	ADC->CCR|=ADC_CCR_TSVREFE;
	
	//Select channel 17 as input (Vrefint)
	ADC1->SQR3|=ADC_SQR3_SQ1_0|ADC_SQR3_SQ1_4;
	
	//Enable analog Watchdog Interrupt
	//AWCD watching channel 17 (Vrefint)
	ADC1->CR1=ADC_CR1_AWDSGL|ADC_CR1_AWDIE|ADC_CR1_AWDEN|ADC_CR1_AWDCH_0|ADC_CR1_AWDCH_4;
		

	//Set up the threshold
	//These threshold values may differe from one board to anothr!
	ADC1->LTR=(uint16_t)0x0695;
	
	//HTR should be larger than LTR
	ADC1->HTR=(uint16_t)0x0780;
	
	//NVIC configuration
	__enable_irq();
	NVIC_SetPriority(ADC_IRQn,0);
	NVIC_ClearPendingIRQ(ADC_IRQn);
	NVIC_EnableIRQ(ADC_IRQn);
	
}

/*----------------------------------------------------------------------------
  Start the ADC conversion and return the reading
 *----------------------------------------------------------------------------*/
uint16_t Get_ADC1(){

	//Software trigger the conversion
	ADC1->CR2|=ADC_CR2_SWSTART;
	
	//Wait for the completion of the conversion
	while(!(ADC1->SR&(1UL<<1))){}
		
	//Return the reading value	
	return ADC1->DR;

}

/*----------------------------------------------------------------------------
  Start the ADC conversion and return the averaged value of reading
 *----------------------------------------------------------------------------*/
uint16_t Get_Average_ADC1(uint8_t times){
	
	uint8_t t;
	uint32_t temp=0;
	
	for(t=0;t<times;t++){
		temp+=Get_ADC1();
	}
	return temp/times;
}


/*----------------------------------------------------------------------------
  Main is always busy.
 *----------------------------------------------------------------------------*/
int main(void)
{	

	Init_LEDs();
	Init_ADC1();
	
	//Have to wait until the power to stabilize
	//otherwise will trigger the analog watchdog wrongly
	Delay(0x1FFFF);
	
	//Trigger once will be enough for the continous mode
	ADC1->CR2|=ADC_CR2_SWSTART;
	
	//Begin to watch
	Turn_On_LED(Green);
	
	while(1){
	//If the oragne LED is fleshing, it is working properly
	//Can be something else while the analog watchdog is working
	Turn_On_Off_LED(Orange);
	}
}


/*----------------------------------------------------------------------------
  ADC handler, do the last minute's job
 *----------------------------------------------------------------------------*/
void ADC_IRQHandler(void){
	
	//Clear the EXTI pending bits
	NVIC_ClearPendingIRQ(ADC_IRQn);
	
	//If the interrupt is caused by analog watchdog

 	if((ADC1->SR&1<<0)==1){
	//Clear the AWD event flag
	ADC1->SR&=~(ADC_SR_AWD);
	
	//Low Power warning!
	Turn_On_Off_LED(Red);
	
	/*
	Do the last minute's job here
	*/
		
 	}

}


// *******************************ARM University Program Copyright  ARM Ltd 2013*************************************   
