
#include "STM32F4xx.h"


//Decide the delay here
#define delay_for_on 0x03FFFFF
#define delay_for_off 0x01FFFFF

#define MASK(x) (1UL << (x))

#define threshold1 3000
#define threshold2 2000
#define threshold3 1500
#define threshold4 1000

/*------------------------------------------------------------------------------
	Example for initialising 4 on board LEDs
	*------------------------------------------------------------------------------*/
void Init_LEDs(void){
		
		//Enable clock to port D		
		RCC->AHB1ENR|=RCC_AHB1ENR_GPIODEN;

	
		//Configure the pins which are conenected to the on board LEDs
		//to be output
		GPIOD->MODER|=(GPIO_MODER_MODER12_0|//Green
									 GPIO_MODER_MODER13_0|//Orange
									 GPIO_MODER_MODER14_0|//Red
									 GPIO_MODER_MODER15_0);//Blue

}

/*----------------------------------------------------------------------------
  Delay Function
 *----------------------------------------------------------------------------*/
void Delay (uint32_t nCount)
{
	while(nCount--)
	{
			__NOP();
	}
}


/*----------------------------------------------------------------------------
  Intializing the ADC1 with PortA pin1 as input (12-bit precision)
 *----------------------------------------------------------------------------*/
void Init_ADC1(void){
	
	//Enable the clock for ADC module and GPIO Port A
	RCC->AHB1ENR|=RCC_AHB1ENR_GPIOAEN;
	RCC->APB2ENR|=RCC_APB2ENR_ADC1EN;
  
	//Configure the Port A pin 1 to be the Analogue
	GPIOA->MODER|=GPIO_MODER_MODER1;
	GPIOA->PUPDR&=~(GPIO_PUPDR_PUPDR1);
	
	//Set the prescaler for the clock
	RCC->CFGR|=RCC_CFGR_PPRE2_DIV2;
	ADC->CCR|=ADC_CCR_ADCPRE_0;
	

	//Power up the ADC module
	ADC1->CR2|=ADC_CR2_ADON;
	
	//480	cycles, better accuracy than 3 cycles
	ADC1->SMPR1|=ADC_SMPR1_SMP16;
	
	//Select channel 1 as input 
	ADC1->SQR3&=~(ADC_SQR3_SQ1);
	ADC1->SQR3|=ADC_SQR3_SQ1_0;

	
	
}


/*----------------------------------------------------------------------------
  Start the ADC conversion and return the reading
 *----------------------------------------------------------------------------*/
uint16_t Get_ADC1(){

	//Software trigger the conversion
	ADC1->CR2|=ADC_CR2_SWSTART;
	
	//Wait for the completion of the conversion
	while(!(ADC1->SR&1<<1)){}
		
	//Return the reading value	
	return ADC1->DR;

}

/*----------------------------------------------------------------------------
  Start the ADC conversion and return the averaged value of reading
	You can also specify the delay between each conversion
 *----------------------------------------------------------------------------*/
uint16_t Get_Average_ADC1_with_delay(uint8_t times,uint32_t nCount)
{ uint8_t t;
	uint32_t temp=0;
	
	for(t=0;t<times;t++){
		temp+=Get_ADC1();
		Delay(nCount);
	}
	return temp/times;

}

/*----------------------------------------------------------------------------
  Intializing the DAC2 as 12-bit right alignment DAC
	And the corresponding GPIO
 *----------------------------------------------------------------------------*/
void Init_DAC2(void){
	
	//Enable the clock for DAC module and GPIO Port A
	RCC->AHB1ENR|=RCC_AHB1ENR_GPIOAEN;
	RCC->APB1ENR|=RCC_APB1ENR_DACEN;
	
	//Configure the PoRT A pin 5 to be the Analogue
	GPIOA->MODER|=GPIO_MODER_MODER5;
	GPIOA->PUPDR&=~(GPIO_PUPDR_PUPDR5);

	//Enable the DAC
	DAC->CR|=DAC_CR_EN2;
	
	//Write to the DAC channel1 12-bit right-aligned data holding register
	DAC->DHR12R2=0;
	
}

/*----------------------------------------------------------------------------
  Intializing the IRLED, Port B pin 15
 *----------------------------------------------------------------------------*/
void Init_IRLED(void){
		
		//Enable clock to port B		
		RCC->AHB1ENR|=RCC_AHB1ENR_GPIOBEN;

	
		//Configure the pins which are conenected to the on board LEDs
		//to be output
		GPIOB->MODER|=(GPIO_MODER_MODER15_0);

}

/*----------------------------------------------------------------------------
  When output low, the IRLED is turned on
 *----------------------------------------------------------------------------*/
void Turn_IRLED_On() {
		GPIOB->BSRRH|=(1<<15);
}

/*----------------------------------------------------------------------------
  When output high, the IRLED is turned off
 *----------------------------------------------------------------------------*/
void Turn_IRLED_Off() {
		GPIOB->BSRRL|=(1<<15);
}

/*----------------------------------------------------------------------------
  Turn on different LEDs according to the difference of the input voltages
 *----------------------------------------------------------------------------*/

void LED_indicator(uint16_t voltage_diff){
	
		if (voltage_diff>threshold1){
			GPIOD->BSRRH=(MASK(15));
			GPIOD->BSRRH=(MASK(12));
			GPIOD->BSRRH=(MASK(13));
			GPIOD->BSRRL=(MASK(14));
		}
		else if (voltage_diff>threshold2){
			GPIOD->BSRRH=(MASK(15));
			GPIOD->BSRRH=(MASK(12));
			GPIOD->BSRRL=(MASK(13));
			GPIOD->BSRRH=(MASK(14));
		}
		else if (voltage_diff>threshold3){
			GPIOD->BSRRH=(MASK(15));
			GPIOD->BSRRL=(MASK(12));
			GPIOD->BSRRH=(MASK(13));
			GPIOD->BSRRH=(MASK(14));
		}
		else {
			GPIOD->BSRRL=(MASK(15));
			GPIOD->BSRRH=(MASK(12));
			GPIOD->BSRRH=(MASK(13));
			GPIOD->BSRRH=(MASK(14));
		}	
	}




/*----------------------------------------------------------------------------
	Main function will collect data both when the IRLED is on and off
	The compare and multiply the difference and output it to DAC
 *----------------------------------------------------------------------------*/
int main(void)
{
	
	uint16_t voltage_reading_when_off;
	uint16_t voltage_reading_when_on;
  uint16_t voltage_diff;
	
	Init_DAC2();
	Init_ADC1();
	Init_IRLED();
	Turn_IRLED_Off();
	Init_LEDs();
	
	
	while(1){
		//Turn off IRLED and wait
		Turn_IRLED_Off();
		Delay(delay_for_off);
		//Collect data
		voltage_reading_when_off=Get_Average_ADC1_with_delay(20,0);	
		
		
		//Turn on IRLED and wait
		Turn_IRLED_On();
		Delay(delay_for_on);
		//Collect data
		voltage_reading_when_on=Get_Average_ADC1_with_delay(20,0);	
		

		//calculate the voltage difference 
		voltage_diff=voltage_reading_when_off-voltage_reading_when_on;
		
		//light LED indicator
		LED_indicator(voltage_diff);

		//Can monitor the voltage using the DAC or LCD display
		//Typical output range is 1v to 2v
		//When the object gets closer, the output should be higer	
		DAC->DHR12R2=(voltage_reading_when_off-voltage_reading_when_on)*100;	
		
	}

}


// *******************************ARM University Program Copyright  ARM Ltd 2013*************************************   
