﻿/*
*******************************************************************************
  Copyright (C), 2012-2014, Embest Tech. Co., Ltd.
  FileName      : Uart_Interrupt.c
  Author        : lichy       
  Version       : 1.0.0     
  Date          : 2012/04/26
  Description   : This example describes how to using UART in interrupt mode
  Function List : 
  History       :
  <author>      : lichy       
  <time>        : 2012/04/26
  <version>     : 1.0.0 	
  <desc>        : build this moudle				 
*******************************************************************************
*/

/* Includes ------------------------------------------------------------------*/
#include "LPC177x_8x.h"
#include "lpc177x_8x_uart.h"
#include "lpc177x_8x_pinsel.h"
#include "debug_frmwrk.h"

/** @defgroup UART_Interrupt	UART Interrupt
 * @ingroup UART_Examples
 * @{
 */


/************************** PRIVATE DEFINTIONS *************************/
#define   UART_TEST_NUM                  0

#if       (UART_TEST_NUM == 0)
#define   _LPC_UART                      (LPC_UART_TypeDef *)LPC_UART0
#define   _UART_IRQ                      UART0_IRQn
#define   _UART_IRQHander                UART0_IRQHandler
#elif     (UART_TEST_NUM == 1)
#define   _LPC_UART                      (LPC_UART_TypeDef *)LPC_UART1
#define   _UART_IRQ                      UART1_IRQn
#define   _UART_IRQHander                UART1_IRQHandler
#elif     (UART_TEST_NUM == 4)
#define   _LPC_UART                      (LPC_UART_TypeDef *)LPC_UART4
#define   _UART_IRQ                      UART4_IRQn
#define   _UART_IRQHander                UART4_IRQHandler
#elif     (UART_TEST_NUM == 3)
#define   _LPC_UART                      LPC_UART3
#define   _UART_IRQ                      UART3_IRQn
#define   _UART_IRQHander                UART3_IRQHandler
#endif

/* buffer size definition */
#define UART_RING_BUFSIZE 256

/* Buf mask */
#define __BUF_MASK (UART_RING_BUFSIZE-1)
/* Check buf is full or not */
#define __BUF_IS_FULL(head, tail) ((tail&__BUF_MASK)==((head+1)&__BUF_MASK))
/* Check buf will be full in next receiving or not */
#define __BUF_WILL_FULL(head, tail) ((tail&__BUF_MASK)==((head+2)&__BUF_MASK))
/* Check buf is empty */
#define __BUF_IS_EMPTY(head, tail) ((head&__BUF_MASK)==(tail&__BUF_MASK))
/* Reset buf */
#define __BUF_RESET(bufidx)   (bufidx=0)
#define __BUF_INCR(bufidx)    (bufidx=(bufidx+1)&__BUF_MASK)


/************************** PRIVATE TYPES *************************/
/** @brief UART Ring buffer structure */
typedef struct
{
  __IO uint32_t tx_head;                /*!< UART Tx ring buffer head index */
  __IO uint32_t tx_tail;                /*!< UART Tx ring buffer tail index */
  __IO uint32_t rx_head;                /*!< UART Rx ring buffer head index */
  __IO uint32_t rx_tail;                /*!< UART Rx ring buffer tail index */
  __IO uint8_t  tx[UART_RING_BUFSIZE];  /*!< UART Tx data ring buffer */
  __IO uint8_t  rx[UART_RING_BUFSIZE];  /*!< UART Rx data ring buffer */
} UART_RING_BUFFER_T;


/************************** PRIVATE VARIABLES *************************/


/* UART Ring buffer*/
UART_RING_BUFFER_T rb;

/* Current Tx Interrupt enable state */
__IO FlagStatus TxIntStat;
const unsigned char FP_Pack_Head[] = "0xEF,0x01,0xFF,0xFF,0xFF,0xFF";                                               //protcol header 
const unsigned char FP_VfyPwd[10] = {0x01,0x00,0x07,0x13,0x11,0x11,0x11,0x11,0x00,0x5f};   
const unsigned char FP_Get_Img[6] = {0x01,0x00,0x03,0x01,0x0,0x05};                                                  //get/record fingerprint image
const unsigned char FP_Templete_Num[6] ={0x01,0x00,0x03,0x1D,0x00,0x21 };                                            //get total number of template
const unsigned char FP_Search[11]={0x01,0x0,0x08,0x04,0x01,0x0,0x0,0x03,0xA1,0x0,0xB2};                              //search range of fingeprints 0 - 929
const unsigned char FP_Search_0_9[11]={0x01,0x0,0x08,0x04,0x01,0x0,0x0,0x0,0x13,0x0,0x21};                           //search fingerprint no.0-9
unsigned char FP_Img_To_Buffer1[7]={0x01,0x0,0x04,0x02,0x01,0x0,0x08};                                               //save image to BUFFER1
unsigned char FP_Img_To_Buffer2[7]={0x01,0x0,0x04,0x02,0x02,0x0,0x09};                                               //save image to BUFFER2
unsigned char FP_Reg_Model[6]={0x01,0x0,0x03,0x05,0x0,0x09};                                                         //Combine BUFFER1 and BUFFER2 to generate feature template
unsigned char FP_Delet_All_Model[6]={0x01,0x0,0x03,0x0d,0x00,0x11};                                                  //clean up all templates in fingerprint module
unsigned char FP_Reg_Match[6]={0x01,0x00,0x03,0x03,0x00,0x07};                                                       //compare charbuffer1 and charbuffer2
unsigned char FP_Reg_Store[9]={0x01,0x00,0x06,0x06,0x02,0x00,0x00,0x00,0x0f};                                        //save charbuffer2 to template  ID0
unsigned char FP_Reg_LoadChar[9]={0x01,0x00,0x06,0x07,0x02,0x00,0x00,0x00,0x10};                                     //read  template ID0 to charbuffer2
volatile unsigned char  FP_Save_Finger[9]={0x01,0x00,0x06,0x06,0x01,0x00,0x0B,0x00,0x19};                            //save the feature of BUFFER1 to specified position 
volatile unsigned char FP_Delete_Model[10]={0x01,0x00,0x07,0x0C,0x0,0x0,0x0,0x1,0x0,0x0};                            //delete specified template 
//volatile unsigned char FINGER_NUM;


/************************** PRIVATE FUNCTIONS *************************/
/* Interrupt service routines */
void _UART_IRQHander(void);
void UART_IntErr(uint8_t bLSErrType);
void UART_IntTransmit(void);
void UART_IntReceive(void);

uint32_t UARTReceive(LPC_UART_TypeDef *UARTPort, uint8_t *rxbuf, uint32_t buflen);
uint32_t UARTSend(LPC_UART_TypeDef *UARTPort, uint8_t *txbuf, uint32_t buflen);
void print_menu(void);

/*----------------- INTERRUPT SERVICE ROUTINES --------------------------*/
/*********************************************************************//**
 * @brief           UART0 interrupt handler sub-routine
 * @param[in]       None
 * @return          None
 **********************************************************************/
void _UART_IRQHander(void)
{
  uint32_t intsrc, tmp, tmp1;

  /* Determine the interrupt source */
  intsrc = UART_GetIntId(_LPC_UART);
  tmp = intsrc & UART_IIR_INTID_MASK;

  /* Receive Line Status */
  if (tmp == UART_IIR_INTID_RLS) {
    /* Check line status */
    tmp1 = UART_GetLineStatus(_LPC_UART);
    /* Mask out the Receive Ready and Transmit Holding empty status*/
    tmp1 &= (UART_LSR_OE | UART_LSR_PE | UART_LSR_FE \
             | UART_LSR_BI | UART_LSR_RXFE);
    /* If any error exist */
    if (tmp1) {
      UART_IntErr(tmp1);
    }
  }

  /* Receive Data Available or Character time-out	*/
  if ((tmp == UART_IIR_INTID_RDA) || (tmp == UART_IIR_INTID_CTI)) {
    UART_IntReceive();
  }

  /* Transmit Holding Empty */
  if (tmp == UART_IIR_INTID_THRE) {
    UART_IntTransmit();
  }

}

/********************************************************************//**
 * @brief           UART receive function (ring buffer used)
 * @param[in]       None
 * @return          None
 *********************************************************************/
void UART_IntReceive(void)
{
  uint8_t tmpc;
  uint32_t rLen;

  while (1)
  {
    /* Call UART read function in UART driver */
    rLen = UART_Receive(_LPC_UART, &tmpc, 1, NONE_BLOCKING);
    /* If data received*/
    if (rLen) {
      /* Check if buffer is more space
       * If no more space, remaining character will be trimmed out
       */
      if (!__BUF_IS_FULL(rb.rx_head,rb.rx_tail)) {
        rb.rx[rb.rx_head] = tmpc;
        __BUF_INCR(rb.rx_head);
				_DBG("   Displaying received data		");
				_DBH(rb.rx[rb.rx_head]);
				
      }
    } else {
      /* no more data	*/    
      break;
    }
  }
}

/********************************************************************//**
 * @brief           UART transmit function (ring buffer used)
 * @param[in]       None
 * @return          None
 *********************************************************************/
void UART_IntTransmit(void)
{
  /* Disable THRE interrupt */
  UART_IntConfig(_LPC_UART, UART_INTCFG_THRE, DISABLE);

  /* Wait for FIFO buffer empty, transfer UART_TX_FIFO_SIZE bytes
   * of data or break whenever ring buffers are empty */
  /* Wait until THR empty */
  while (UART_CheckBusy(_LPC_UART) == SET);

  while (!__BUF_IS_EMPTY(rb.tx_head,rb.tx_tail))
  {
    /* Move a piece of data into the transmit FIFO */
    if (UART_Send(_LPC_UART, (uint8_t *)&rb.tx[rb.tx_tail], 1, NONE_BLOCKING)) {
      /* Update transmit ring FIFO tail pointer */
      __BUF_INCR(rb.tx_tail);
    } else {
      break;
    }
  }

  /* If there is no more data to send, disable the transmit
     interrupt - else enable it or keep it enabled */
  if (__BUF_IS_EMPTY(rb.tx_head, rb.tx_tail)) {
    UART_IntConfig(_LPC_UART, UART_INTCFG_THRE, DISABLE);
    /* Reset Tx Interrupt state*/
    TxIntStat = RESET;
  } else {
    /* Set Tx Interrupt state */
    TxIntStat = SET;
    UART_IntConfig(_LPC_UART, UART_INTCFG_THRE, ENABLE);
  }
}


/*********************************************************************//**
 * @brief          UART Line Status Error
 * @param[in]       bLSErrType        UART Line Status Error Type
 * @return          None
 **********************************************************************/
void UART_IntErr(uint8_t bLSErrType)
{
  /* Loop forever*/
  while (1)
  {
    /* For testing purpose */
  }
}

/*-------------------------PRIVATE FUNCTIONS------------------------------*/
/*********************************************************************//**
 * @brief           UART transmit function for interrupt mode (using ring buffers)
 * @param[in]       UARTPort	Selected UART peripheral used to send data,
 *                  should be UART0
 * @param[out]      txbuf Pointer to Transmit buffer
 * @param[in]       buflen Length of Transmit buffer
 * @return          Number of bytes actually sent to the ring buffer
 **********************************************************************/
uint32_t UARTSend(LPC_UART_TypeDef *UARTPort, uint8_t *txbuf, uint32_t buflen)
{
  uint8_t *data = (uint8_t *) txbuf;
  uint32_t bytes = 0;

  /* Temporarily lock out UART transmit interrupts during this
     read so the UART transmit interrupt won't cause problems
     with the index values */
  UART_IntConfig(UARTPort, UART_INTCFG_THRE, DISABLE);

  /* Loop until transmit run buffer is full or until n_bytes
     expires */
  while ((buflen > 0) && (!__BUF_IS_FULL(rb.tx_head, rb.tx_tail)))
  {
    /* Write data from buffer into ring buffer */
    rb.tx[rb.tx_head] = *data;
    data++;

    /* Increment head pointer */
    __BUF_INCR(rb.tx_head);

    /* Increment data count and decrement buffer size count */
    bytes++;
    buflen--;
  }

  /*
   * Check if current Tx interrupt enable is reset, 
   * that means the Tx interrupt must be re-enabled
   * due to call UART_IntTransmit() function to trigger
   * this interrupt type
   */
  if (TxIntStat == RESET) {
    UART_IntTransmit();
  } else {
    /*
     * Otherwise, re-enables Tx Interrupt
     */  
    UART_IntConfig(UARTPort, UART_INTCFG_THRE, ENABLE);
  }

  return bytes;
}


/*********************************************************************//**
 * @brief           UART read function for interrupt mode (using ring buffers)
 * @param[in]       UARTPort  Selected UART peripheral used to send data,
 *                  should be UART0
 * @param[out]      rxbuf Pointer to Received buffer
 * @param[in]       buflen Length of Received buffer
 * @return          Number of bytes actually read from the ring buffer
 **********************************************************************/
uint32_t UARTReceive(LPC_UART_TypeDef *UARTPort, uint8_t *rxbuf, uint32_t buflen)
{
  uint8_t *data = (uint8_t *) rxbuf;
  uint32_t bytes = 0;

  /* Temporarily lock out UART receive interrupts during this
     read so the UART receive interrupt won't cause problems
     with the index values */
  UART_IntConfig(UARTPort, UART_INTCFG_RBR, DISABLE);

  /* Loop until receive buffer ring is empty or
     until max_bytes expires */
  while ((buflen > 0) && (!(__BUF_IS_EMPTY(rb.rx_head, rb.rx_tail))))
  {
    /* Read data from ring buffer into user buffer */
    *data = rb.rx[rb.rx_tail];
    data++;

    /* Update tail pointer */
    __BUF_INCR(rb.rx_tail);
    
    /* Increment data count and decrement buffer size count */
    bytes++;
    buflen--;
  }

  /* Re-enable UART interrupts */
  UART_IntConfig(UARTPort, UART_INTCFG_RBR, ENABLE);

  return bytes;
}

/*********************************************************************//**
 * @brief           Print Welcome Screen Menu subroutine
 * @param           None
 * @return          None
 **********************************************************************/
void print_menu(void)
{
  uint32_t tmp, tmp2;
  uint8_t *pDat,i;

  tmp = sizeof (FP_Pack_Head);
  tmp2 = 0;
	_DBD(tmp);
	_DBG(" ");
  pDat = (uint8_t *)&FP_Pack_Head[0];
  if (tmp>0)
  {
		for(i=0;i<16;i++){
			_DBG(" ");
		UARTPutHex(LPC_UART0,FP_Pack_Head[i]);
			_DBG(" in hex = ");
			_DBH(FP_Pack_Head[i]);
		}
	}
//	tmp = sizeof (FP_VfyPwd);
//		pDat = (uint8_t *)& FP_VfyPwd[0];
//		 if(tmp>0)
//  {
//    tmp2 = UARTSend(_LPC_UART, pDat,tmp);

//		}
//	
  
}

/*-------------------------MAIN FUNCTION------------------------------*/
/*********************************************************************//**
 * @brief           c_entry: Main UART program body
 * @param[in]       None
 * @return          None
 **********************************************************************/
void c_entry(void)
{
  /* UART Configuration structure variable */
  UART_CFG_Type UARTConfigStruct;
  /* UART FIFO configuration Struct variable */
  UART_FIFO_CFG_Type UARTFIFOConfigStruct;

  uint32_t idx, len; 
  FlagStatus exitflag;
  uint8_t buffer[10];
#if (UART_TEST_NUM == 0)
  /*
   * Initialize UART0 pin connect
   * P0.2: U0_TXD
   * P0.3: U0_RXD
   */
  PINSEL_ConfigPin(0,2,1);
  PINSEL_ConfigPin(0,3,1);
#elif (UART_TEST_NUM == 1)
  /*
   * Initialize UART1 pin connect
   * P3.16: U1_TXD
   * P3.17: U1_RXD
   */
  PINSEL_ConfigPin(3,16,3);
  PINSEL_ConfigPin(3,17,3);
#elif (UART_TEST_NUM == 4)
  /*
   * Initialize UART4 pin connect
   * P5.4: U4_TXD
   * P5.3: U3_RXD
   */
  PINSEL_ConfigPin(5,4,4);
  PINSEL_ConfigPin(5,3,4);
#elif (UART_TEST_NUM == 3)
  /*
   * Initialize UART3 pin connect
   * P0.0: U3_TXD
   * P0.1: U3_RXD
   */
  PINSEL_ConfigPin(0,0,2);
  PINSEL_ConfigPin(0,1,2);
#endif


  /* Initialize UART Configuration parameter structure to default state:
   * Baudrate = 115200 bps
   * 8 data bit
   * 1 Stop bit
   * None parity
   */
	 debug_frmwrk_init();

  UART_ConfigStructInit(&UARTConfigStruct);

  /* Initialize UART0 peripheral with given to corresponding parameter */
  UART_Init(LPC_UART0, &UARTConfigStruct);


  /* Initialize FIFOConfigStruct to default state:
   *  - FIFO_DMAMode = DISABLE
   *  - FIFO_Level = UART_FIFO_TRGLEV0
   *  - FIFO_ResetRxBuf = ENABLE
   *  - FIFO_ResetTxBuf = ENABLE
   *  - FIFO_State = ENABLE
   */
  UART_FIFOConfigStructInit(&UARTFIFOConfigStruct);

  /* Initialize FIFO for UART0 peripheral*/
  UART_FIFOConfig((LPC_UART_TypeDef *)_LPC_UART, &UARTFIFOConfigStruct);


  /* Enable UART Transmit*/
  UART_TxCmd((LPC_UART_TypeDef *)_LPC_UART, ENABLE);

  /* Enable UART Rx interrupt */
  UART_IntConfig((LPC_UART_TypeDef *)_LPC_UART, UART_INTCFG_RBR, ENABLE);
  /* Enable UART line status interrupt */
  UART_IntConfig((LPC_UART_TypeDef *)_LPC_UART, UART_INTCFG_RLS, ENABLE);
  /*
   * Do not enable transmit interrupt here, since it is handled by
   * UART_Send() function, just to reset Tx Interrupt state for the
   * first time
   */
  TxIntStat = RESET;

  /* Reset ring buf head and tail idx*/
  __BUF_RESET(rb.rx_head);
  __BUF_RESET(rb.rx_tail);
  __BUF_RESET(rb.tx_head);
  __BUF_RESET(rb.tx_tail);

  /* preemption = 1, sub-priority = 1 */
  NVIC_SetPriority(_UART_IRQ, ((0x01 << 3) | 0x01));
  
  /* Enable Interrupt for UART0 channel */
  NVIC_EnableIRQ(_UART_IRQ);


  /* print welcome screen*/
  print_menu();

  /* reset exit flag */
  exitflag = RESET;

  /* Read some data from the buffer */
  while (exitflag == RESET)
  {
    len = 0;
    while (len == 0)
    {
      len = UARTReceive(_LPC_UART, buffer, sizeof(buffer));
    }

    /* Got some data */
    idx = 0;
    while (idx < len)
    {
      
      idx++;
    }
  }

  /* wait for current transmission complete - THR must be empty */
  while (UART_CheckBusy(_LPC_UART));

  /* DeInitialize UART0 peripheral */
  UART_DeInit(_LPC_UART);

  /* Loop forever */
  while (1);
}

/* With ARM and GHS toolsets, the entry point is main() - this will
   allow the linker to generate wrapper code to setup stacks, allocate
   heap area, and initialize and copy code and data segments. For GNU
   toolsets, the entry point is through __start() in the crt0_gnu.asm
   file, and that startup code will setup stacks and data */
int main(void)
{
  c_entry();
  return 0;
}



/**
 * @}
 */
