/////**************************************************************************//**
//// * @file     FlashPrg.c
//// * @brief    Flash Programming Functions adapted for New Device Flash
//// * @version  V1.0.0
//// * @date     10. January 2018
//// ******************************************************************************/
/////*
//// * Copyright (c) 2010-2018 Arm Limited. All rights reserved.
//// *
//// * SPDX-License-Identifier: Apache-2.0
//// *
//// * Licensed under the Apache License, Version 2.0 (the License); you may
//// * not use this file except in compliance with the License.
//// * You may obtain a copy of the License at
//// *
//// * www.apache.org/licenses/LICENSE-2.0
//// *
//// * Unless required by applicable law or agreed to in writing, software
//// * distributed under the License is distributed on an AS IS BASIS, WITHOUT
//// * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
//// * See the License for the specific language governing permissions and
//// * limitations under the License.
//// */
//// 
#include "..\FlashOS.H"        // FlashOS Structures

#include <stdint.h>
#include <stddef.h> 
//#include "FlashOS.h" /* Keil header available in MDK packs */
static inline void _alg_disable_irq(void){ __asm volatile ("cpsid i"); }
static inline void _alg_enable_irq(void){ __asm volatile ("cpsie i"); }

/********************** User Configuration ************************/
#ifndef FLEXSPI1_BASE
#define FLEXSPI1_BASE   (0x400CC000u)   /* FlexSPI1 registers (adjust if needed) */
#endif
#define FLASH_AHB_BASE  (0x30000000u)   /* Your mapped XIP base */
#define FLASH_TOTAL_SZ  (0x08000000u)   /* 128MB (1Gb) */

/********************** ISSI Opcodes/Consts ************************/
#define CMD_RDID        0x9F
#define CMD_WREN        0x06
#define CMD_RDSR        0x05
#define CMD_READ        0x03
#define CMD_PP          0x02
#define CMD_SE_4K       0x20
#define CMD_BE_64K      0xD8
#define CMD_CE          0xC7
/* Bank Address Register (BAR)  volatile */
#define CMD_RDBR_16     0x16
#define CMD_WRBRV_17    0x17

#define JEDEC_MFG       0x9D
#define JEDEC_MEMTYPE   0x60
#define JEDEC_CAP       0x21

#define PAGE_SIZE       256u
#define SECTOR_SIZE     0x1000u

/********************** Minimal FlexSPI Reg Map *********************/
/* This is a reduced subset sufficient for IP commands and LUT handling. */
typedef struct {
  volatile uint32_t MCR0;       // 0x000
  volatile uint32_t MCR1;       // 0x004
  volatile uint32_t MCR2;       // 0x008
  volatile uint32_t AHBCR;      // 0x00C
  volatile uint32_t INTEN;      // 0x010
  volatile uint32_t INTR;       // 0x014
  volatile uint32_t LUTKEY;     // 0x018
  volatile uint32_t LUTCR;      // 0x01C
  volatile uint32_t AHBRXBUF0CR0; // 0x020
  volatile uint32_t AHBRXBUF1CR0; // 0x024
  volatile uint32_t AHBRXBUF2CR0; // 0x028
  volatile uint32_t AHBRXBUF3CR0; // 0x02C
  volatile uint32_t AHBRXBUF4CR0; // 0x030
  volatile uint32_t AHBRXBUF5CR0; // 0x034
  volatile uint32_t AHBRXBUF6CR0; // 0x038
  volatile uint32_t AHBRXBUF7CR0; // 0x03C
  uint32_t RESERVED0[24];       // 0x040..0x09C
  volatile uint32_t FLSHCR0[4]; // 0x0A0..0x0AC
  volatile uint32_t FLSHCR1[4]; // 0x0B0..0x0BC
  volatile uint32_t FLSHCR2[4]; // 0x0C0..0x0CC
  volatile uint32_t FLSHCR4;    // 0x0D0
  uint32_t RESERVED1[8];        // 0x0D4..0x0F0
  volatile uint32_t IPCR0;      // 0x0F4 Serial Flash Address
  volatile uint32_t IPCR1;      // 0x0F8 SeqID/SeqNum + DataSize
  volatile uint32_t IPCR2;      // 0x0FC (if present; not always used)
  volatile uint32_t IPCMD;      // 0x100 Trigger
  volatile uint32_t IPRXFCR;    // 0x104 RX FIFO control
  volatile uint32_t IPTXFCR;    // 0x108 TX FIFO control
  volatile uint32_t DLLCR[2];   // 0x10C..
  uint32_t RESERVED2[26];       // up to 0x180
  volatile uint32_t TXDATAn;    // 0x180 write data FIFO (word)  write repeatedly
  volatile uint32_t RXDATAn;    // 0x184 read data FIFO (word)  read repeatedly
  uint32_t RESERVED3[70];       // ...
  volatile uint32_t LUT[64];    // 0x200.. LUT (64 x 32-bit)
} FLEXSPI_Regs;

#define FLEXSPI1   ((FLEXSPI_Regs *)FLEXSPI1_BASE)

/* Bits/keys */
#define MCR0_MDIS           (1u<<1)
#define INTR_IPCMDDONE      (1u<<1)
#define INTR_IPRXWA         (1u<<2)
#define INTR_IPTXWE         (1u<<3)
#define INTR_IPRXWATER      INTR_IPRXWA
#define INTR_IPTXWATER      INTR_IPTXWE
#define LUTKEY_VALUE        (0x5AF05AF0u)
#define LUTCR_UNLOCK        (0x02u)
#define LUTCR_LOCK          (0x01u)

/* IPCR1 helpers: layout differs per RT; we encode as: [SEQID:0..5][SEQNUM:8..10][IDATSZ:16..31] */
#define IPCR1_SEQID(x)      ((x) & 0x3Fu)
#define IPCR1_SEQNUM(x)     (((x) & 0x7u) << 8)
#define IPCR1_IDATSZ(x)     (((x) & 0xFFFFu) << 16)

/* LUT encoding helpers (simplified) */
#define LUT_OP(op,pads,imm)   ( ((op)&0x3F) | (((pads)&3)<<8) | (((imm)&0xFF)<<10) )
#define LUT_CMD(c)            LUT_OP(0x01,0,c)
#define LUT_ADDR(bits)        LUT_OP(0x02,0,(bits))
#define LUT_READ(len)         LUT_OP(0x09,0,(len))
#define LUT_WRITE(len)        LUT_OP(0x08,0,(len))
#define LUT_STOP()            LUT_OP(0x00,0,0)

/******************* 3-Byte LUT (24-bit) *******************/
enum { L_WREN=0, L_RDSR=4, L_RDID=8, L_READ=12, L_PP=16, L_SE4K=20, L_BE64=24, L_CE=28, L_RDBR=32, L_WRBR=36 };
static const uint32_t LUT_3B[] = {
  /* L_WREN */  LUT_CMD(CMD_WREN), LUT_STOP(), 0, 0,
  /* L_RDSR */  LUT_CMD(CMD_RDSR), LUT_READ(1), 0, 0,
  /* L_RDID */  LUT_CMD(CMD_RDID), LUT_READ(3), 0, 0,
  /* L_READ */  LUT_CMD(CMD_READ), LUT_ADDR(24), LUT_READ(0), LUT_STOP(),
  /* L_PP   */  LUT_CMD(CMD_PP),   LUT_ADDR(24), LUT_WRITE(0), LUT_STOP(),
  /* L_SE4K */  LUT_CMD(CMD_SE_4K),LUT_ADDR(24), 0, 0,
  /* L_BE64 */  LUT_CMD(CMD_BE_64K),LUT_ADDR(24),0, 0,
  /* L_CE   */  LUT_CMD(CMD_CE),   LUT_STOP(),   0, 0,
  /* L_RDBR */  LUT_CMD(CMD_RDBR_16), LUT_READ(1), 0, 0,
  /* L_WRBR */  LUT_CMD(CMD_WRBRV_17), LUT_WRITE(1), 0, 0,
};

/******************* Low-level helpers *********************/
static void flexspi_wait_ip_done(void) {
  while (!(FLEXSPI1->INTR & INTR_IPCMDDONE)) { /* spin */ }
  FLEXSPI1->INTR = INTR_IPCMDDONE; /* clear */
}

static void flexspi_load_lut(const uint32_t *lut, unsigned words) {
  FLEXSPI1->MCR0 |= MCR0_MDIS;                 /* disable */
  FLEXSPI1->LUTKEY = LUTKEY_VALUE;
  FLEXSPI1->LUTCR  = LUTCR_UNLOCK;
  for (unsigned i=0; i<words && i<64; ++i) {
    FLEXSPI1->LUT[i] = lut[i];
  }
  FLEXSPI1->LUTCR  = LUTCR_LOCK;
  FLEXSPI1->MCR0  &= ~MCR0_MDIS;               /* enable */
}

/* Issue IP command sequence. If tx!=NULL -> write; if rx!=NULL -> read. len in bytes. */
static void flexspi_ip_xfer(uint8_t seqIdx, uint32_t addr, void *buf, uint32_t len, int is_read) {
  /* Clear FIFOs */
  FLEXSPI1->IPRXFCR = 1; /* RX FIFO clear */
  FLEXSPI1->IPTXFCR = 1; /* TX FIFO clear */

  FLEXSPI1->IPCR0 = addr;                                  /* serial flash address (lower 24b used) */
  FLEXSPI1->IPCR1 = IPCR1_SEQID(seqIdx) | IPCR1_SEQNUM(1) | IPCR1_IDATSZ(len);

  if (!is_read && buf && len) {
    /* push TX data as words */
    const uint8_t *p = (const uint8_t*)buf;
    for (uint32_t i=0;i<len;){
      uint32_t w=0; unsigned n=0;
      for (; n<4 && i<len; ++n,++i) w |= ((uint32_t)p[i]) << (8*n);
      FLEXSPI1->TXDATAn = w;
    }
  }

  FLEXSPI1->IPCMD = 1;                                      /* trigger */
  flexspi_wait_ip_done();

  if (is_read && buf && len) {
    uint8_t *p = (uint8_t*)buf;
    for (uint32_t i=0;i<len;){
      uint32_t w = FLEXSPI1->RXDATAn;
      for (unsigned n=0;n<4 && i<len;++n,++i) p[i] = (uint8_t)(w >> (8*n));
    }
  }
}

static void wren(void){ flexspi_ip_xfer(L_WREN, 0, NULL, 0, 0); }
static uint8_t rdsr(void){ uint8_t sr=0; flexspi_ip_xfer(L_RDSR,0,&sr,1,1); return sr; }
static int wait_wip(uint32_t ms){ while (ms--){ if ((rdsr() & 1u)==0) return 0; } return 1; }

/* Compute BAR from absolute address. BANK = bits [26:24] for 128MB range. */
static int set_bank_for_addr(uint32_t absAdr){
  uint8_t bank = (uint8_t)((absAdr - FLASH_AHB_BASE) >> 24); /* 0..7 */
  uint8_t cur=0; flexspi_ip_xfer(L_RDBR,0,&cur,1,1);
  if ((cur & 0x07) == bank) return 0;
  wren(); flexspi_ip_xfer(L_WRBR,0,&bank,1,0);
  return 0;
}

/*********************** FlashOS API ************************/
int Init (unsigned long adr, unsigned long clk, unsigned long fnc) {
  (void)adr; (void)clk; (void)fnc;

  /* ---------------- Safety: disable IRQs to avoid ISR/XIP bus conflicts ---------------- */
  __asm volatile ("cpsid i");   // mask all interrupts while flash ops are active

  /* ---------------- Optional: disable watchdogs (if enabled in startup) ---------------- */
  // Example for RT117x: disable WDOG3 and RTWDOG (if needed)
  // WDOG3->WCR &= ~WDOG_WCR_WDE_MASK;
  // RTWDOG->CS = (RTWDOG->CS & ~RTWDOG_CS_EN_MASK) | RTWDOG_CS_UPDATE_MASK;

  /* ---------------- Minimal FlexSPI bring-up ---------------- */
  // Ensure clock and pad mux are configured externally (boot code).
  // For safety, disable AHB prefetch/caching during programming.
  FLEXSPI1->AHBCR = 0x00000000u;   // disable AHB prefetch/caching

  /* ---------------- Clear and load LUT table (3-byte ops) ---------------- */
  for (unsigned i = 0; i < 64; i++)
    FLEXSPI1->LUT[i] = 0;
  flexspi_load_lut(LUT_3B, sizeof(LUT_3B)/sizeof(uint32_t));

  /* ---------------- Verify JEDEC ID (optional enforcement) ---------------- */
  uint8_t id[3] = {0};
  flexspi_ip_xfer(L_RDID, 0, id, 3, 1);
  // Uncomment below lines to enforce strict JEDEC ID match
  // if (id[0] != JEDEC_MFG || id[1] != JEDEC_MEMTYPE || id[2] != JEDEC_CAP) {
  //   __asm volatile ("cpsie i");   // re-enable IRQs before returning
  //   return 1;
  // }

  /* ---------------- Set Bank Register to 0 (start window 0x0000000xFFFFFF) ---------------- */
  if (set_bank_for_addr(FLASH_AHB_BASE)) {
    __asm volatile ("cpsie i");
    return 1;
  }

  /* ---------------- Re-enable IRQs after safe init ---------------- */
  __asm volatile ("cpsie i");

  return 0;   // success
}


int UnInit (unsigned long fnc) { (void)fnc; return 0; }

int EraseChip(void) {
  _alg_disable_irq();
  wren();
  flexspi_ip_xfer(L_CE, 0, NULL, 0, 0);
  int rc = wait_wip(100000);   // 100 s
  _alg_enable_irq();
  return rc;
}

int EraseSector (unsigned long adr) {
  _alg_disable_irq();
  if (set_bank_for_addr((uint32_t)adr)) { _alg_enable_irq(); return 1; }
  wren();
  flexspi_ip_xfer(L_SE4K, (uint32_t)adr, NULL, 0, 0);
  int rc = wait_wip(6000);
  _alg_enable_irq();
  return rc;
}

int ProgramPage (unsigned long adr, unsigned long sz, unsigned char *buf) {
  while (sz){
    unsigned page_off = (unsigned)(adr & (PAGE_SIZE-1));
    unsigned chunk = PAGE_SIZE - page_off; if (chunk>sz) chunk=(unsigned)sz;
    if (set_bank_for_addr((uint32_t)adr)) return 1;
    wren();
    flexspi_ip_xfer(L_PP, (uint32_t)adr, buf, chunk, 0);
    if (wait_wip(1000)) return 1;
    adr += chunk; buf += chunk; sz -= chunk;
  }
  return 0;
}

/*********************** FlashDevice ************************/
/* Use EXTSPI device type; sector layout uses offsets (not absolute addresses). */

/* Integration notes:
 * - If your FlexSPI1 register base differs, update FLEXSPI1_BASE.
 * - Ensure clocks/pinmux for FlexSPI1 and pads are configured by your startup.
 * - This algorithm uses single-SPI (1-1-1) ops for maximum compatibility.
 * - For speed, you can add Fast Read (0x0B) with dummy cycles and Quad Enable.
 */

