# ###############################################################################
#
# Copyright (C) 2024 Arm Limited (or its affiliates). All rights reserved.
#
# Zephyr RTOS OS Awareness Extension for ARM Development Studio
#
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

from osapi import *
from utils import *
from globs import *

# Timers class
class Timers( Table ):

    # Class ID
    ID = "timers"

    # Column definitions
    cols = \
    [
        [ TIMER_EXPIRY_FN,      FORMAT_ADDRESS_STR  ],
        [ TIMER_STOP_FN,        FORMAT_ADDRESS_STR  ],
        [ TIMER_PERIOD,         FORMAT_NUMBER_STR   ],
        [ TIMER_STATUS,         FORMAT_NUMBER_STR   ],
        [ TIMER_USER_DATA,      FORMAT_ADDRESS_STR  ],
    ]

    # Class constructor
    def __init__( self ):

        # Class ID
        cid = self.ID

        # Create primary field
        fields = [ createPrimaryField( cid, K_TIMER, TEXT ) ]

        # Create timer columns
        fields.append( createField( cid, "expiryFn", TEXT ) )
        fields.append( createField( cid, "stopFn", TEXT ) )
        fields.append( createField( cid, "period", TEXT ) )
        fields.append( createField( cid, "status", TEXT ) )
        fields.append( createField( cid, "userData", TEXT ) )

        # Create table
        Table.__init__( self, cid, fields )

    # Read timer
    def readTimer( self, cid, members, debugSession ):

        # Populate primary field
        cells = [ createTextCell( cid ) ]

        # Get expiry function
        expiryFn = "N/A"
        if TIMER_EXPIRY_FN in members:
            addr = members[ TIMER_EXPIRY_FN ].readAsNumber( )
            if addr != 0:
                expiryFn = hex( addr )
        cells.append( createTextCell( expiryFn ) )

        # Get stop function
        stopFn = "N/A"
        if TIMER_STOP_FN in members:
            addr = members[ TIMER_STOP_FN ].readAsNumber( )
            if addr != 0:
                stopFn = hex( addr )
        cells.append( createTextCell( stopFn ) )

        # Get period
        period = "N/A"
        if TIMER_PERIOD in members:
            period = str( members[ TIMER_PERIOD ].readAsNumber( ) )
        cells.append( createTextCell( period ) )

        # Get status
        status = "N/A"
        if TIMER_STATUS in members:
            statusVal = members[ TIMER_STATUS ].readAsNumber( )
            if statusVal == 0:
                status = "Inactive"
            else:
                status = str( statusVal ) + " expirations"
        cells.append( createTextCell( status ) )

        # Get user data
        userData = "N/A"
        if TIMER_USER_DATA in members:
            addr = members[ TIMER_USER_DATA ].readAsNumber( )
            if addr != 0:
                userData = hex( addr )
        cells.append( createTextCell( userData ) )

        # Populated record
        return self.createRecord( cells )

    # Get timers
    def getRecords( self, debugSession ):

        # Blank records
        records = [ ]

        # Zephyr doesn't have a central registry
        try:
            # Look for timer symbols
            timerSymbols = self.findTimerSymbols( debugSession )

            for timerSymbol in timerSymbols:
                try:
                    timerExpr = debugSession.evaluateExpression( timerSymbol )
                    timerId = hex( addressExprsToLong( timerExpr ) )
                    timerMembers = timerExpr.getStructureMembers( )

                    # Verify this is a valid timer
                    if TIMER_EXPIRY_FN in timerMembers or TIMER_PERIOD in timerMembers:
                        records.append( self.readTimer( timerId, timerMembers, debugSession ) )
                except:
                    pass
        except:
            pass

        return records

    # Find timer symbols
    def findTimerSymbols( self, debugSession ):
        commonTimerNames = [
            "app_timer",
            "sys_timer",
            "work_timer",
            "timeout_timer",
            "poll_timer",
        ]

        foundTimers = []
        for name in commonTimerNames:
            if debugSession.symbolExists( name ):
                foundTimers.append( name )

        return foundTimers
