# ###############################################################################
#
# Copyright (C) 2024 Arm Limited (or its affiliates). All rights reserved.
#
# Zephyr RTOS OS Awareness Extension for ARM Development Studio
#
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

from osapi import *
from utils import *
from globs import *

# Semaphores class
class Semaphores( Table ):

    # Class ID
    ID = "semaphores"

    # Column definitions
    cols = \
    [
        [ SEM_COUNT,    FORMAT_NUMBER_STR,  TEXT    ],
        [ SEM_LIMIT,    FORMAT_NUMBER_STR,  TEXT    ],
    ]

    # Class constructor
    def __init__( self ):

        # Class ID
        cid = self.ID

        # Create primary field
        fields = [ createPrimaryField( cid, K_SEM, TEXT ) ]

        # Create semaphore columns
        fields.append( createField( cid, "count", TEXT ) )
        fields.append( createField( cid, "limit", TEXT ) )
        fields.append( createField( cid, "waiters", TEXT ) )

        # Create table
        Table.__init__( self, cid, fields )

    # Read semaphore
    def readSemaphore( self, cid, members, debugSession ):

        # Populate primary field
        cells = [ createTextCell( cid ) ]

        # Get count
        count = "0"
        if SEM_COUNT in members:
            count = str( members[ SEM_COUNT ].readAsNumber( ) )
        cells.append( createTextCell( count ) )

        # Get limit
        limit = "N/A"
        if SEM_LIMIT in members:
            limit = str( members[ SEM_LIMIT ].readAsNumber( ) )
        cells.append( createTextCell( limit ) )

        # Get waiters
        waiters = "None"
        if SEM_WAIT_Q in members:
            waitQ = members[ SEM_WAIT_Q ]
            waiterList = readWaitQueue( debugSession, waitQ )
            if len( waiterList ) > 0:
                waiters = getThreadNamesFromList( waiterList, debugSession )
        cells.append( createTextCell( waiters ) )

        # Populated record
        return self.createRecord( cells )

    # Get semaphores
    def getRecords( self, debugSession ):

        # Blank records
        records = [ ]

        # Zephyr doesn't have a central registry of semaphores
        # We need to search for k_sem structures in the symbol table

        try:
            # Look for semaphore symbols
            semSymbols = self.findSemaphoreSymbols( debugSession )

            for semSymbol in semSymbols:
                try:
                    semExpr = debugSession.evaluateExpression( semSymbol )
                    semId = hex( addressExprsToLong( semExpr ) )
                    semMembers = semExpr.getStructureMembers( )

                    # Verify this is a valid semaphore by checking for expected members
                    if SEM_COUNT in semMembers:
                        records.append( self.readSemaphore( semId, semMembers, debugSession ) )
                except:
                    pass
        except:
            pass

        return records

    # Find semaphore symbols in the debug session
    def findSemaphoreSymbols( self, debugSession ):
        # Try some common patterns
        commonSemNames = [
            "done_sem",
            "sync_sem",
            "tx_sem",
            "rx_sem",
            "ready_sem",
        ]

        foundSems = []
        for name in commonSemNames:
            if debugSession.symbolExists( name ):
                foundSems.append( name )

        return foundSems
