# ###############################################################################
#
# Copyright (C) 2024 Arm Limited (or its affiliates). All rights reserved.
#
# Zephyr RTOS OS Awareness Extension for ARM Development Studio
#
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

from osapi import *
from utils import *
from globs import *

# Mutexes class
class Mutexes( Table ):

    # Class ID
    ID = "mutexes"

    # Column definitions
    cols = \
    [
        [ MUTEX_OWNER,              FORMAT_ADDRESS_STR, TEXT    ],
        [ MUTEX_LOCK_COUNT,         FORMAT_NUMBER_STR,  TEXT    ],
        [ MUTEX_OWNER_ORIG_PRIO,    FORMAT_NUMBER_STR,  TEXT    ],
    ]

    # Class constructor
    def __init__( self ):

        # Class ID
        cid = self.ID

        # Create primary field
        fields = [ createPrimaryField( cid, K_MUTEX, TEXT ) ]

        # Create mutex columns
        fields.append( createField( cid, "owner", TEXT ) )
        fields.append( createField( cid, "ownerName", TEXT ) )
        fields.append( createField( cid, "lockCount", TEXT ) )
        fields.append( createField( cid, "origPriority", TEXT ) )
        fields.append( createField( cid, "waiters", TEXT ) )

        # Create table
        Table.__init__( self, cid, fields )

    # Read mutex
    def readMutex( self, cid, members, debugSession ):

        # Populate primary field
        cells = [ createTextCell( cid ) ]

        # Get owner address
        ownerAddr = "N/A"
        ownerName = "None"
        if MUTEX_OWNER in members:
            ownerPtr = members[ MUTEX_OWNER ]
            if ownerPtr.readAsNumber( ) != 0:
                ownerAddr = hex( ownerPtr.readAsNumber( ) )
                # Try to get owner thread name
                try:
                    ownerThread = ownerPtr.dereferencePointer( )
                    ownerMembers = ownerThread.getStructureMembers( )
                    if THREAD_NAME in ownerMembers:
                        namePtr = ownerMembers[ THREAD_NAME ]
                        if namePtr.readAsNumber( ) != 0:
                            ownerName = namePtr.readAsNullTerminatedString( )
                except:
                    pass

        cells.append( createTextCell( ownerAddr ) )
        cells.append( createTextCell( ownerName ) )

        # Get lock count
        lockCount = "0"
        if MUTEX_LOCK_COUNT in members:
            lockCount = str( members[ MUTEX_LOCK_COUNT ].readAsNumber( ) )
        cells.append( createTextCell( lockCount ) )

        # Get original priority
        origPrio = "N/A"
        if MUTEX_OWNER_ORIG_PRIO in members:
            prio = members[ MUTEX_OWNER_ORIG_PRIO ].readAsNumber( )
            if prio > 127:
                prio = prio - 256
            origPrio = str( prio )
        cells.append( createTextCell( origPrio ) )

        # Get waiters
        waiters = "None"
        if MUTEX_WAIT_Q in members:
            waitQ = members[ MUTEX_WAIT_Q ]
            waiterList = readWaitQueue( debugSession, waitQ )
            if len( waiterList ) > 0:
                waiters = getThreadNamesFromList( waiterList, debugSession )
        cells.append( createTextCell( waiters ) )

        # Populated record
        return self.createRecord( cells )

    # Get mutexes
    def getRecords( self, debugSession ):

        # Blank records
        records = [ ]

        # Zephyr doesn't have a central registry of mutexes like FreeRTOS
        # We need to search for k_mutex structures in the symbol table
        # This is typically done by looking for global mutex variables

        # Try to find mutex symbols
        try:
            # Look for common mutex naming patterns
            # This is a limitation - we can only find globally defined mutexes
            mutexSymbols = self.findMutexSymbols( debugSession )

            for mutexSymbol in mutexSymbols:
                try:
                    mutexExpr = debugSession.evaluateExpression( mutexSymbol )
                    mutexId = hex( addressExprsToLong( mutexExpr ) )
                    mutexMembers = mutexExpr.getStructureMembers( )

                    # Verify this is a valid mutex by checking for expected members
                    if MUTEX_LOCK_COUNT in mutexMembers:
                        records.append( self.readMutex( mutexId, mutexMembers, debugSession ) )
                except:
                    pass
        except:
            pass

        return records

    # Find mutex symbols in the debug session
    def findMutexSymbols( self, debugSession ):
        # This is a placeholder - in practice, you would need to:
        # 1. Use debugSession.getSymbols() if available
        # 2. Or maintain a list of known mutex names
        # 3. Or use RTOS-specific debugging features

        # For now, try some common patterns
        commonMutexNames = [
            "sys_mutex",
            "console_mutex",
            "printk_mutex",
            "log_mutex",
        ]

        foundMutexes = []
        for name in commonMutexNames:
            if debugSession.symbolExists( name ):
                foundMutexes.append( name )

        return foundMutexes
