# ###############################################################################
#
# Copyright (C) 2024 Arm Limited (or its affiliates). All rights reserved.
#
# Zephyr RTOS OS Awareness Extension for ARM Development Studio
#
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

from osapi import *
from utils import *
from globs import *

# Message Queues class
class MessageQueues( Table ):

    # Class ID
    ID = "msgqs"

    # Column definitions
    cols = \
    [
        [ MSGQ_MSG_SIZE,        FORMAT_NUMBER_STR,  TEXT    ],
        [ MSGQ_MAX_MSGS,        FORMAT_NUMBER_STR,  TEXT    ],
        [ MSGQ_USED_MSGS,       FORMAT_NUMBER_STR,  TEXT    ],
        [ MSGQ_BUFFER_START,    FORMAT_ADDRESS_STR, TEXT    ],
        [ MSGQ_BUFFER_END,      FORMAT_ADDRESS_STR, TEXT    ],
        [ MSGQ_READ_PTR,        FORMAT_ADDRESS_STR, TEXT    ],
        [ MSGQ_WRITE_PTR,       FORMAT_ADDRESS_STR, TEXT    ],
    ]

    # Class constructor
    def __init__( self ):

        # Class ID
        cid = self.ID

        # Create primary field
        fields = [ createPrimaryField( cid, K_MSGQ, TEXT ) ]

        # Create message queue columns
        fields.append( createField( cid, "msgSize", TEXT ) )
        fields.append( createField( cid, "maxMsgs", TEXT ) )
        fields.append( createField( cid, "usedMsgs", TEXT ) )
        fields.append( createField( cid, "bufferStart", TEXT ) )
        fields.append( createField( cid, "bufferEnd", TEXT ) )
        fields.append( createField( cid, "readPtr", TEXT ) )
        fields.append( createField( cid, "writePtr", TEXT ) )
        fields.append( createField( cid, "waiters", TEXT ) )

        # Create table
        Table.__init__( self, cid, fields )

    # Read message queue
    def readMsgQ( self, cid, members, debugSession ):

        # Populate primary field
        cells = [ createTextCell( cid ) ]

        # Get message size
        msgSize = "N/A"
        if MSGQ_MSG_SIZE in members:
            msgSize = str( members[ MSGQ_MSG_SIZE ].readAsNumber( ) )
        cells.append( createTextCell( msgSize ) )

        # Get max messages
        maxMsgs = "N/A"
        if MSGQ_MAX_MSGS in members:
            maxMsgs = str( members[ MSGQ_MAX_MSGS ].readAsNumber( ) )
        cells.append( createTextCell( maxMsgs ) )

        # Get used messages
        usedMsgs = "0"
        if MSGQ_USED_MSGS in members:
            usedMsgs = str( members[ MSGQ_USED_MSGS ].readAsNumber( ) )
        cells.append( createTextCell( usedMsgs ) )

        # Get buffer start
        bufferStart = "N/A"
        if MSGQ_BUFFER_START in members:
            bufferStart = hex( members[ MSGQ_BUFFER_START ].readAsNumber( ) )
        cells.append( createTextCell( bufferStart ) )

        # Get buffer end
        bufferEnd = "N/A"
        if MSGQ_BUFFER_END in members:
            bufferEnd = hex( members[ MSGQ_BUFFER_END ].readAsNumber( ) )
        cells.append( createTextCell( bufferEnd ) )

        # Get read pointer
        readPtr = "N/A"
        if MSGQ_READ_PTR in members:
            readPtr = hex( members[ MSGQ_READ_PTR ].readAsNumber( ) )
        cells.append( createTextCell( readPtr ) )

        # Get write pointer
        writePtr = "N/A"
        if MSGQ_WRITE_PTR in members:
            writePtr = hex( members[ MSGQ_WRITE_PTR ].readAsNumber( ) )
        cells.append( createTextCell( writePtr ) )

        # Get waiters
        waiters = "None"
        if MSGQ_WAIT_Q in members:
            waitQ = members[ MSGQ_WAIT_Q ]
            waiterList = readWaitQueue( debugSession, waitQ )
            if len( waiterList ) > 0:
                waiters = getThreadNamesFromList( waiterList, debugSession )
        cells.append( createTextCell( waiters ) )

        # Populated record
        return self.createRecord( cells )

    # Get message queues
    def getRecords( self, debugSession ):

        # Blank records
        records = [ ]

        # Zephyr doesn't have a central registry of message queues
        try:
            # Look for message queue symbols
            msgqSymbols = self.findMsgQSymbols( debugSession )

            for msgqSymbol in msgqSymbols:
                try:
                    msgqExpr = debugSession.evaluateExpression( msgqSymbol )
                    msgqId = hex( addressExprsToLong( msgqExpr ) )
                    msgqMembers = msgqExpr.getStructureMembers( )

                    # Verify this is a valid message queue
                    if MSGQ_MSG_SIZE in msgqMembers or MSGQ_MAX_MSGS in msgqMembers:
                        records.append( self.readMsgQ( msgqId, msgqMembers, debugSession ) )
                except:
                    pass
        except:
            pass

        return records

    # Find message queue symbols
    def findMsgQSymbols( self, debugSession ):
        commonMsgQNames = [
            "cmd_msgq",
            "data_msgq",
            "event_msgq",
            "sensor_msgq",
        ]

        foundMsgQs = []
        for name in commonMsgQNames:
            if debugSession.symbolExists( name ):
                foundMsgQs.append( name )

        return foundMsgQs
