# ###############################################################################
#
# Copyright (C) 2024 Arm Limited (or its affiliates). All rights reserved.
#
# Zephyr RTOS OS Awareness Extension for ARM Development Studio
#
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

# Import all!
from osapi import *
from utils import *
from globs import *

# Table headings (defined in messages.properties)
ITEM_HEADING    = "item"
VALUE_HEADING   = "value"
DESC_HEADING    = "desc"

# Kernel global variables and information
KERNEL_ITEMS = \
[
    [ [ Z_KERNEL, Z_KERNEL_CURRENT        ], FORMAT_ADDRESS_STR, "Pointer to currently running thread"                    ],
    [ [ Z_KERNEL, Z_KERNEL_THREADS        ], FORMAT_ADDRESS_STR, "Pointer to head of all threads linked list"             ],
    [ [ Z_KERNEL, Z_KERNEL_READY_Q        ], FORMAT_ADDRESS_STR, "Ready queue structure"                                  ],
    [ [ Z_KERNEL, Z_KERNEL_UPTIME_TICKS   ], FORMAT_NUMBER_STR,  "System uptime in ticks"                                 ],
    [ [ KERNEL_VERSION_STRING             ], FORMAT_STRING,      "Zephyr kernel version string"                           ],
]

# Kernel data
class KernelData( Table ):

    # Constructor
    def __init__( self ):

        # Id name
        tid = "kernel"

        # Configure headings
        fields = \
        [
            createField( tid, ITEM_HEADING,  TEXT ),
            createField( tid, VALUE_HEADING, TEXT ),
            createField( tid, DESC_HEADING,  TEXT )
        ]

        # Create table
        Table.__init__( self, tid, fields )

    # Read table entry
    def readRecord( self, itemIndex, debugSession ):

        # Get item details
        itemRefs   = KERNEL_ITEMS[ itemIndex ][ 0 ]      # Item reference
        itemType   = KERNEL_ITEMS[ itemIndex ][ 1 ]      # Item format type
        itemDesc   = KERNEL_ITEMS[ itemIndex ][ 2 ]      # Item description

        # Create reference to be evaluated
        itemRef = globCreateRef( itemRefs )

        # Make sure reference exists?
        if debugSession.symbolExists( itemRefs[0] ):

            try:
                # Evaluate expression
                expr = debugSession.evaluateExpression( itemRef )

                # Format expression
                itemValue = formatExpr( expr, itemType, debugSession, "" )

                # Value created?
                if len( str(itemValue) ):

                    # Yes, create entry in table
                    cells = [ createTextCell( itemRef ), createTextCell( str(itemValue) ), createTextCell( itemDesc ) ]

                    # Create record entry
                    return self.createRecord( cells )
            except:
                pass

    # Get all kernel configuration data
    def getRecords( self, debugSession ):

        # Clear records
        records = [ ]

        # Get maximum number of items
        j = len( KERNEL_ITEMS )

        # Get each item
        for i in range( j ):
            record = self.readRecord( i, debugSession )
            if record:
                records.append( record )

        # Add Zephyr-specific kernel information
        self.addZephyrInfo( records, debugSession )

        # All items
        return records

    # Add Zephyr-specific information
    def addZephyrInfo( self, records, debugSession ):

        # Try to get version information
        try:
            if debugSession.symbolExists( "sys_kernel_version_get" ):
                # Version is typically stored or can be obtained
                pass

            # Get thread count
            if debugSession.symbolExists( Z_KERNEL ):
                kernelExpr = debugSession.evaluateExpression( Z_KERNEL )
                members = kernelExpr.getStructureMembers( )

                # Count threads
                if Z_KERNEL_THREADS in members:
                    threadsPtr = members[ Z_KERNEL_THREADS ]
                    if threadsPtr.readAsNumber( ):
                        threadItems = readThreadItems( debugSession, threadsPtr )
                        threadCount = len( threadItems )
                        cells = [ createTextCell( "Thread Count" ), createTextCell( str(threadCount) ), createTextCell( "Total number of threads" ) ]
                        records.append( self.createRecord( cells ) )

                # Check if current thread exists
                if Z_KERNEL_CURRENT in members:
                    currentPtr = members[ Z_KERNEL_CURRENT ]
                    if currentPtr.readAsNumber( ):
                        currentThread = currentPtr.dereferencePointer( )
                        threadMembers = currentThread.getStructureMembers( )
                        if THREAD_NAME in threadMembers:
                            namePtr = threadMembers[ THREAD_NAME ]
                            if namePtr.readAsNumber( ):
                                name = namePtr.readAsNullTerminatedString( )
                                cells = [ createTextCell( "Current Thread" ), createTextCell( name ), createTextCell( "Currently executing thread" ) ]
                                records.append( self.createRecord( cells ) )
        except:
            pass
