# ###############################################################################
#
# Copyright (C) 2024 Arm Limited (or its affiliates). All rights reserved.
#
# Zephyr RTOS OS Awareness Extension for ARM Development Studio
#
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

# Import all!
from osapi import *

# General definitions
OS_NAME                                 = "zephyr"

# Zephyr Kernel Global Variables
Z_KERNEL                                = "_kernel"
Z_KERNEL_CURRENT                        = "current"
Z_KERNEL_THREADS                        = "threads"
Z_KERNEL_READY_Q                        = "ready_q"
Z_KERNEL_IDLE_THREADS                   = "idle_threads"
Z_KERNEL_UPTIME_TICKS                   = "uptime"

# Zephyr Version Information
KERNEL_VERSION_NUMBER                   = "KERNEL_VERSION_NUMBER"
KERNEL_VERSION_STRING                   = "KERNEL_VERSION_STRING"

# Thread States (base.thread_state bitmask)
THREAD_STATE_DUMMY                      = 0x01  # Thread is dummy
THREAD_STATE_PENDING                    = 0x02  # Thread is pending
THREAD_STATE_PRESTART                   = 0x04  # Thread is pre-started
THREAD_STATE_DEAD                       = 0x08  # Thread is dead
THREAD_STATE_SUSPENDED                  = 0x10  # Thread is suspended
THREAD_STATE_ABORTING                   = 0x20  # Thread is aborting
THREAD_STATE_QUEUED                     = 0x80  # Thread is queued

# Thread State Names
THREAD_STATES = \
{
    0x00 : "READY",
    0x01 : "DUMMY",
    0x02 : "PENDING",
    0x04 : "PRESTART",
    0x08 : "DEAD",
    0x10 : "SUSPENDED",
    0x20 : "ABORTING",
    0x80 : "QUEUED"
}

# Zephyr Structure Names
K_THREAD                                = "k_thread"
K_THREAD_BASE                           = "_thread_base"
K_THREAD_STACK_INFO                     = "_thread_stack_info"
K_SEM                                   = "k_sem"
K_MUTEX                                 = "k_mutex"
K_MSGQ                                  = "k_msgq"
K_FIFO                                  = "k_fifo"
K_LIFO                                  = "k_lifo"
K_STACK                                 = "k_stack"
K_TIMER                                 = "k_timer"
K_WORK                                  = "k_work"
K_WORK_Q                                = "k_work_q"
SYS_DLIST                               = "sys_dlist_t"
SYS_SLIST                               = "sys_slist_t"

# Thread structure members
THREAD_BASE                             = "base"
THREAD_CALLEE_SAVED                     = "callee_saved"
THREAD_STACK_INFO                       = "stack_info"
THREAD_NEXT_THREAD                      = "next_thread"
THREAD_PRIO                             = "prio"
THREAD_SCHED_LOCKED                     = "sched_locked"
THREAD_USER_OPTIONS                     = "user_options"
THREAD_THREAD_STATE                     = "thread_state"
THREAD_SWAP_DATA                        = "swap_data"
THREAD_NAME                             = "name"
THREAD_INIT_DATA                        = "init_data"
THREAD_JOIN_WAITERS                     = "join_waiters"
THREAD_TIMEOUT                          = "timeout"
THREAD_SWITCH_HANDLE                    = "switch_handle"

# Thread base structure members
THREAD_BASE_PRIO                        = "prio"
THREAD_BASE_SCHED_LOCKED                = "sched_locked"
THREAD_BASE_USER_OPTIONS                = "user_options"
THREAD_BASE_THREAD_STATE                = "thread_state"

# Thread stack info structure members
STACK_INFO_START                        = "start"
STACK_INFO_SIZE                         = "size"
STACK_INFO_DELTA                        = "delta"

# Callee-saved registers (ARM Cortex-M)
CALLEE_SAVED_V1                         = "v1"  # R4
CALLEE_SAVED_V2                         = "v2"  # R5
CALLEE_SAVED_V3                         = "v3"  # R6
CALLEE_SAVED_V4                         = "v4"  # R7
CALLEE_SAVED_V5                         = "v5"  # R8
CALLEE_SAVED_V6                         = "v6"  # R9
CALLEE_SAVED_V7                         = "v7"  # R10
CALLEE_SAVED_V8                         = "v8"  # R11
CALLEE_SAVED_PSP                        = "psp" # Process Stack Pointer

# Semaphore structure members
SEM_COUNT                               = "count"
SEM_LIMIT                               = "limit"
SEM_WAIT_Q                              = "wait_q"

# Mutex structure members
MUTEX_OWNER                             = "owner"
MUTEX_LOCK_COUNT                        = "lock_count"
MUTEX_OWNER_ORIG_PRIO                   = "owner_orig_prio"
MUTEX_WAIT_Q                            = "wait_q"

# Message Queue structure members
MSGQ_MSG_SIZE                           = "msg_size"
MSGQ_MAX_MSGS                           = "max_msgs"
MSGQ_BUFFER_START                       = "buffer_start"
MSGQ_BUFFER_END                         = "buffer_end"
MSGQ_READ_PTR                           = "read_ptr"
MSGQ_WRITE_PTR                          = "write_ptr"
MSGQ_USED_MSGS                          = "used_msgs"
MSGQ_WAIT_Q                             = "wait_q"
MSGQ_LOCK                               = "lock"

# FIFO/LIFO structure members (uses k_queue internally)
QUEUE_DATA_Q                            = "data_q"
QUEUE_WAIT_Q                            = "wait_q"

# Timer structure members
TIMER_TIMEOUT                           = "timeout"
TIMER_EXPIRY_FN                         = "expiry_fn"
TIMER_STOP_FN                           = "stop_fn"
TIMER_PERIOD                            = "period"
TIMER_STATUS                            = "status"
TIMER_USER_DATA                         = "user_data"

# Work structure members
WORK_NODE                               = "node"
WORK_HANDLER                            = "handler"

# Work Queue structure members
WORKQ_THREAD                            = "thread"
WORKQ_QUEUE                             = "queue"

# Doubly-linked list structure members
DLIST_HEAD                              = "head"
DLIST_TAIL                              = "tail"

# Singly-linked list structure members
SLIST_HEAD                              = "head"
SLIST_TAIL                              = "tail"

# Wait queue structure members
WAITQ_WAITQ                             = "waitq"

# Register map names
REG_MAP_V8MBASIC                        = "v8MBasic"
REG_MAP_V8MVFP                          = "v8MVFP"
REG_MAP_V7MBASIC                        = "v7MBasic"
REG_MAP_V7MVFP                          = "v7MVFP"
REG_MAP_V7ABASIC                        = "v7ABasic"
REG_MAP_V7AVFP                          = "v7AVFP"

# General definitions
YES                                     = 1
NO                                      = 0

# Stack sentinel magic value for overflow detection
STACK_SENTINEL_MAGIC                    = 0xAAAAAAAA

# Create global reference
def globCreateRef( refs ):
    return '.'.join( refs )
