# ###############################################################################
#
# Copyright (C) 2024 Arm Limited (or its affiliates). All rights reserved.
#
# Zephyr RTOS OS Awareness Extension for ARM Development Studio
#
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

from osapi import *
from utils import *
from globs import *

# FIFOs class (also applies to LIFOs as they share similar structure)
class FIFOs( Table ):

    # Class ID
    ID = "fifos"

    # Class constructor
    def __init__( self ):

        # Class ID
        cid = self.ID

        # Create primary field
        fields = [ createPrimaryField( cid, K_FIFO, TEXT ) ]

        # Create FIFO columns
        fields.append( createField( cid, "type", TEXT ) )
        fields.append( createField( cid, "dataQueue", TEXT ) )
        fields.append( createField( cid, "waiters", TEXT ) )

        # Create table
        Table.__init__( self, cid, fields )

    # Read FIFO
    def readFIFO( self, cid, members, fifoType, debugSession ):

        # Populate primary field
        cells = [ createTextCell( cid ) ]

        # Set type
        cells.append( createTextCell( fifoType ) )

        # Get data queue address
        dataQueue = "N/A"
        if QUEUE_DATA_Q in members:
            dataQueue = hex( addressExprsToLong( members[ QUEUE_DATA_Q ] ) )
        cells.append( createTextCell( dataQueue ) )

        # Get waiters
        waiters = "None"
        if QUEUE_WAIT_Q in members:
            waitQ = members[ QUEUE_WAIT_Q ]
            waiterList = readWaitQueue( debugSession, waitQ )
            if len( waiterList ) > 0:
                waiters = getThreadNamesFromList( waiterList, debugSession )
        cells.append( createTextCell( waiters ) )

        # Populated record
        return self.createRecord( cells )

    # Get FIFOs and LIFOs
    def getRecords( self, debugSession ):

        # Blank records
        records = [ ]

        # Zephyr doesn't have a central registry
        try:
            # Look for FIFO symbols
            fifoSymbols = self.findFIFOSymbols( debugSession )

            for fifoSymbol, fifoType in fifoSymbols:
                try:
                    fifoExpr = debugSession.evaluateExpression( fifoSymbol )
                    fifoId = hex( addressExprsToLong( fifoExpr ) )
                    fifoMembers = fifoExpr.getStructureMembers( )

                    records.append( self.readFIFO( fifoId, fifoMembers, fifoType, debugSession ) )
                except:
                    pass
        except:
            pass

        return records

    # Find FIFO and LIFO symbols
    def findFIFOSymbols( self, debugSession ):
        commonFIFONames = [
            ("tx_fifo", "FIFO"),
            ("rx_fifo", "FIFO"),
            ("data_fifo", "FIFO"),
            ("event_fifo", "FIFO"),
            ("tx_lifo", "LIFO"),
            ("rx_lifo", "LIFO"),
            ("free_lifo", "LIFO"),
        ]

        foundFIFOs = []
        for name, fifoType in commonFIFONames:
            if debugSession.symbolExists( name ):
                foundFIFOs.append( (name, fifoType) )

        return foundFIFOs
